//---------------------------------------------------------------------------
// TVMesh.h
//
// Header for self-contained VMesh data (Microsoft Freelancer 3D Model
// format) handler class
//
// Written by Kurt Fitzner <kfitzner@excelcia.org>
// Copyright  2003, Kurt Fitzner
// Revised August 14, 2007, Steve Breslin <steve.breslin@gmail.com>
//---------------------------------------------------------------------------
// This file is part of msCMPImporter.
//
// msCMPImporter is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// msCMPImporter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#ifndef TVMeshH
#define TVMeshH
//---------------------------------------------------------------------------
#ifndef __NOEXCEPT
  #define __NOEXCEPT(stmnt) { try { stmnt; } catch(...) { ; } }
#endif
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Some helper classes first.  TMaterial, TVertex, TTriangle, and TMesh
// along with their TList-derived array handling containers.
//---------------------------------------------------------------------------
class TVMesh;

class TMeshGroup {
  public:
    __fastcall TMeshGroup(void) { Name = ""; StartMesh = MeshCount = StartVertex = EndVertex = StartVertexRef = EndVertexRef = 0; OriginX = OriginY = OriginZ = 0.0f; };

    AnsiString  Name;
    int         StartMesh;
    int         MeshCount;
    int         StartVertex;
    int         EndVertex;
    int         StartVertexRef;
    int         EndVertexRef;
    float       OriginX;
    float       OriginY;
    float       OriginZ;
};  // class TMeshGroup

class TMeshGroups : public TList {
  public:
    inline       __fastcall  TMeshGroups(TVMesh *Owner) : TList() { this->Owner = Owner; };
    inline       __fastcall ~TMeshGroups() { Clear(); };
    TMeshGroup * __fastcall  Extract(TMeshGroup *Item);
    inline int   __fastcall  Add(TMeshGroup *Item) { return TList::Add(Item); };
           int   __fastcall  Add(AnsiString Name, int StartMesh, int MeshCount, int StartVertex, int EndVertex, int StartVertexRef, int EndVertexRef, float OriginX = 0.0f, float OriginY = 0.0f, float OriginZ = 0.0f);
           int   __fastcall  IndexOf(AnsiString Name);
           void  __fastcall  Clear(void) { while (Count) { delete Items[0]; Delete(0); } SetCount(0); SetCapacity(0); };

    __property TMeshGroup   *Items[int Index] = {read = Get, write = Put};
    __property TMeshGroup * *List = {read=FList};
  protected:
    inline TMeshGroup * __fastcall Get(int Index) { return (TMeshGroup *)TList::Get(Index); };
    inline void         __fastcall Put(int Index, TMeshGroup *Item) { TList::Put(Index, Item); };
  private:
    TVMesh * Owner;

    TMeshGroup * *FList;
};  // class TMaterials



class TMaterial {
  public:
    DWORD      MaterialID;
    AnsiString Name;
    AnsiString Library;
    AnsiString FileName;
    int        LibraryNode;
    bool       IsScanned;
};  // class TMaterial

inline static int __fastcall SortCompare(void *Item1, void *Item2) { return (((TMaterial *)Item2)->MaterialID > ((TMaterial *)Item1)->MaterialID)?-1:((((TMaterial *)Item2)->MaterialID < ((TMaterial *)Item1)->MaterialID)?1:0); };

class TMaterials : public TList {
  public:
    inline      __fastcall  TMaterials(TVMesh *Owner) : TList() { this->Owner = Owner; };
    inline      __fastcall ~TMaterials() { Clear(); };
    inline void __fastcall  Sort(void) { TList::Sort(SortCompare); };
    TMaterial * __fastcall  Extract(TMaterial *Item);
    inline int  __fastcall  Add(TMaterial *Item) { int n = TList::Add(Item); Sort(); return n; };
           int  __fastcall  Add(DWORD MaterialID, AnsiString Name, AnsiString Library = "", AnsiString FileName = "", int LibraryNode = -1, bool IsScanned = false);
    inline int  __fastcall  Compare(TMaterial *Item1, TMaterial *Item2) { return SortCompare(Item1, Item2); };
           int  __fastcall  IndexOf(DWORD MaterialID);
    inline int  __fastcall  IndexOf(TMaterial *Material) { return TList::IndexOf(Material); }
           void __fastcall  Clear(void) { while (Count) { delete Items[0]; Delete(0); } SetCount(0); SetCapacity(0); };

    __property TMaterial   *Items[int Index] = {read = Get, write = Put};
    __property TMaterial * *List = {read=FList};
    __property int          ScannedCount = {read = GetScannedCount};
  protected:
    inline TMaterial * __fastcall Get(int Index) { return (TMaterial *)TList::Get(Index); };
    inline void        __fastcall Put(int Index, TMaterial *Item) { TList::Put(Index, Item); };
           int         __fastcall GetScannedCount(void);
  private:
    TVMesh * Owner;

    TMaterial * *FList;
};  // class TMaterials


class TVertex {
  public:
    float X;
    float Y;
    float Z;
    float NormalX;
    float NormalY;
    float NormalZ;
    float U;
    float V;
};  // class TVertex

class TVertices : public TList {
  public:
    inline     __fastcall  TVertices(TVMesh *Owner) : TList() { this->Owner = Owner; };
    inline     __fastcall ~TVertices() { Clear(); };
    inline int __fastcall  Add(TVertex * Item) { return TList::Add(Item); };
    int        __fastcall  Add(float X, float Y, float Z, float NormalX, float NormalY, float NormalZ, float U, float V);
    virtual void __fastcall Clear(void) { while (Count) { delete Items[0]; Delete(0); } SetCount(0); SetCapacity(0); };
    __property TVertex *Items[int Index] = {read = Get, write = Put};
    __property TVertex * *List = {read=FList};
  protected:
    inline TVertex * __fastcall Get(int Index) { return (TVertex *)TList::Get(Index); };
    inline void      __fastcall Put(int Index, TVertex *Item) { TList::Put(Index, Item); };
  private:
    TVMesh *Owner;
    TVertex * *FList;
};  // class TVertices

class TTriangle {
  public:
    int Vertex1;
    int Vertex2;
    int Vertex3;
};  // class TTriangle

class TTriangles : public TList {
  public:
    inline     __fastcall  TTriangles(TVMesh *Owner) : TList() { this->Owner = Owner; };
    inline     __fastcall ~TTriangles() { Clear(); };
    inline int __fastcall  Add(TTriangle * Item) { return TList::Add(Item); };
    int        __fastcall  Add(int Vertex1, int Vertex2, int Vertex3);
    void       __fastcall  Clear(void) { while (Count) { delete Items[0]; Delete(0); } SetCount(0); SetCapacity(0); };
    __property TTriangle *Items[int Index] = {read = Get, write = Put};
    __property TTriangle * *List = {read=FList};
  protected:
    inline TTriangle * __fastcall Get(int Index) { return (TTriangle *)TList::Get(Index); };
    inline void        __fastcall Put(int Index, TTriangle *Item) { TList::Put(Index, Item); };
  private:
    TVMesh *Owner;
    TTriangle * *FList;
};  // class TTriangles

class TMesh;

class TMeshes : public TList {
  public:
    inline     __fastcall  TMeshes(TVMesh *Owner) : TList() { this->Owner = Owner; };
    inline     __fastcall ~TMeshes() { Clear(); };
    inline int __fastcall  Add(TMesh * Item) { return TList::Add(Item); };
    int        __fastcall  Add(DWORD MaterialID, int StartVertex, int EndVertex, int VertexRefCount);
    virtual void __fastcall Clear(void) { while (Count) { delete Items[0]; Delete(0); } SetCount(0); SetCapacity(0); };
    __property TMesh *Items[int Index] = {read = Get, write = Put};
    __property TMesh * *List = {read=FList};
  protected:
    inline TMesh * __fastcall Get(int Index) { return (TMesh *)TList::Get(Index); };
    inline void    __fastcall Put(int Index, TMesh *Item) { TList::Put(Index, Item); };
  private:
    TVMesh  *Owner;
    TMesh * *FList;
};  // class TMeshes

class TMesh {
  public:
    inline __fastcall TMesh(TMeshes *Owner) { this->Owner = Owner; };
    DWORD MaterialID;
    int   RelStartVertex;
    int   RelEndVertex;
    int   VertexRefCount;
    __property int StartVertex   = {read = GetStartVertex};
    __property int EndVertex     = {read = GetEndVertex};
    __property int VertexCount   = {read = GetVertexCount};
    __property int StartTriangle = {read = GetStartTriangle};
    __property int EndTriangle   = {read = GetEndTriangle};
    __property int TriangleCount = {read = GetTriangleCount};
  protected:
           int __fastcall GetStartVertex(void)   { int SV = 0; for (int n=1; n<=Owner->IndexOf(this); n++) if (!Owner->Items[n]->RelStartVertex) SV+=Owner->Items[n-1]->RelEndVertex+1; return SV+RelStartVertex; };
    inline int __fastcall GetEndVertex(void)     { return StartVertex + VertexCount - 1; };
    inline int __fastcall GetVertexCount(void)   { return RelEndVertex - RelStartVertex + 1;}
           int __fastcall GetStartTriangle(void) { int ST = 0; for (int n=0; n<Owner->IndexOf(this); ST+=Owner->Items[n]->TriangleCount,n++); return ST;};
    inline int __fastcall GetEndTriangle(void)   { return GetStartTriangle() + GetTriangleCount() - 1;};
    inline int __fastcall GetTriangleCount(void) { return VertexRefCount/3; };
  private:
    TMeshes *Owner;
};  // class TMesh

//---------------------------------------------------------------------------
// Class TVMesh
//
// Main VMesh handling class.
//---------------------------------------------------------------------------
class TVMesh : public TObject
{
  public:
         __fastcall  TVMesh(void);
         __fastcall ~TVMesh();
    void __fastcall  Clear(void);
    bool __fastcall  LoadFromStream(TStream *InStream, bool Rewind = false);

    inline bool __fastcall IsLoaded(void) { return Meshes->Count && Triangles->Count && Vertices->Count; };

    TMeshGroups *MeshGroups;
    TMaterials  *Materials;
    TMeshes     *Meshes;
    TTriangles  *Triangles;
    TVertices   *Vertices;


  protected:
    //-----------------------------------------------------------------------
    // Some structures used purely internally for loading in a VMesh from
    // a file.  These all represent raw structures as they exist in the
    // VMesh file format.
    //-----------------------------------------------------------------------
    struct TVMeshRawHeader {
      DWORD Signature1;                          // Always 0x00000001 ?? Version number?
      DWORD Signature2;                          // Always 0x00000004 ?? Version number?
      WORD  MeshCount;
      WORD  VertexRefCount;
      WORD  Signature3;                          // Always 0x0112 ?? Breslin note: this is the encoding type. It can also be 212, and maybe some others.
      WORD  VertexCount;
    };  // struct TVMeshRawHeader

    #pragma pack(push, 4)
    struct TVMeshRawMesh {
      DWORD MaterialID;
      WORD  StartVertex;
      WORD  EndVertex;
      WORD  VertexRefCount;
      WORD  Signature1;                          // Always 0x00CC ?? DWORD padding?
    };
    #pragma pack(pop)

    #pragma pack(push, 2)
    struct TVMeshRawTriangle {
      WORD Vertex1;
      WORD Vertex2;
      WORD Vertex3;
    };
    #pragma pack(pop)

    struct TVMeshRawVertex {
      float X;
      float Y;
      float Z;
      float NormalX;
      float NormalY;
      float NormalZ;
      float U;
      float V;

    };

//-- Breslin Changes.

    /* Here I've just added another struct, for decoding 212s.
     *
     * 212s are the same as 112s, but they have have two meaningless integers
     * at the end. These additional integers serve no purpose, so our only
     * task here is to ignore them nicely.
     */
    struct TVMeshRawVertex212 {
      float X;
      float Y;
      float Z;
      float NormalX;
      float NormalY;
      float NormalZ;
      float U;
      float V;

      /* We pick up the two throwaway values, so they don't get put at the top
       * of the next raw read.
       */
      int throwaway1;
      int throwaway2;

    };

//-- End Breslin Changes.

    //-----------------------------------------------------------------------
    // End of the internal raw VMesh structures
    //-----------------------------------------------------------------------

  private:

};  // class TVMesh
//---------------------------------------------------------------------------
#endif
