//---------------------------------------------------------------------------
// msCMPImporter.dll - Utilities.cpp
//
// Miscelaneous stuff - helper functions and usefull classes
//
// Written by Kurt Fitzner <kfitzner@excelcia.org>
// Copyright  2003, Kurt Fitzner
//---------------------------------------------------------------------------
// This file is part of msCMPImporter.
//
// msCMPImporter is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// msCMPImporter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

#include <vcl.h>
#include <windows.h>
#pragma hdrstop

#include "IL\IL.h"
#include "IL\ILU.h"
#include <JPEG.hpp>
#include "TProgramLog.h"

#include "Utilities.h"
//---------------------------------------------------------------------------
#pragma link "DevIL.lib"

//---------------------------------------------------------------------------
// generate_crc()
//
// Calculate a 32-bit crc based on the input string.
// This routine requires a resource to be included in the executable called
// "CRCTABLE" that holds the generator table.  It will load and lock this
// resource the first time the function is called, so no prior initialization
// is required.
//
// Takes:   char pointer pointing to a zstring
// Returns: integer containing crc
// Throws:  EResNotFound if it cannot find the CRCTABLE resource
//          generic Exception if it can't load or lock the CRCTABLE resource
//           or if the CRCTABLE resource is the wrong size (256 DWORDS) 
//---------------------------------------------------------------------------
unsigned int generate_crc (char *string)
{
  __ENTERFUNCTION__;

  static DWORD *crctable = NULL;
  unsigned int crc;
  unsigned len = strlen(string);

  if (!crctable) {
    HRSRC CRCResource;
    HGLOBAL CRCGlobal;

    CRCResource = FindResource(hModuleDLL, "CRCTABLE", RT_RCDATA);
    if (!CRCResource)
      throw EResNotFound("generate_crc() could not find CRCTABLE resource.");
    if (SizeofResource(hModuleDLL, CRCResource) != sizeof(DWORD) * 256)
      throw Exception("generate_crc() CRCTABLE resource is wrong size.");
    CRCGlobal = LoadResource(hModuleDLL, CRCResource);
    if (!CRCGlobal)
      throw Exception("generate_crc(): Could not load CRCTABLE resource.");
    crctable = (DWORD *)LockResource(CRCGlobal);
    if (!crctable)
      throw Exception("generate_crc(): Could not lock CRCTABLE resource");
  }  // if (!crctable)

  crc = 0xFFFFFFFFL;

  for (unsigned n = 0; n < len; n++)
    crc = ((crc>>8) & 0x00FFFFFFL) ^ crctable[ (crc^ (tolower(string[n])) ) & 0xFF ];

  crc = crc^0xFFFFFFFFL;

  __LEAVEFUNCTION__;
  return crc;
}  // unsigned int crc (char *string)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TempFileName()
//
// Make a temporary filename with path.  Uses the current system temp
// folder.
//
// Takes:   AnsiString file extension of temporary file name
// Returns: A full path temp filename
// Throws:  Nothing
//---------------------------------------------------------------------------
AnsiString TempFileName(AnsiString Extension) {
  __ENTERFUNCTION__;
  static int Count = 0;
  char       TempFolder[MAX_PATH];
  AnsiString retval;

  GetTempPath(MAX_PATH, TempFolder);

  retval = IncludeTrailingPathDelimiter(String(TempFolder)) + String("~") + IntToHex((int)GetActiveWindow(),8) + IntToHex(Count++, 8) + Extension;
  __LEAVEFUNCTION__;
  return retval;
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ConvertImage()
//
// Loads a source image in the file format specified by its extension and
// saves it in a new file format specified by the destination extension
//---------------------------------------------------------------------------
bool ConvertImage(AnsiString Source, AnsiString Dest)
{
  __ENTERFUNCTION__;
  if (ExtractFileExt(Dest) != ".JPG") {
    ILuint ImgId;

    ilInit();
    ilGenImages(1, &ImgId);
    ilBindImage(ImgId);
    if (!ilLoadImage(Source.c_str()))
      return false;
    ilEnable(IL_FILE_OVERWRITE);
    ilSaveImage(Dest.c_str());
    ilDeleteImages(1, &ImgId);
  }  // if (ExtractFileExt(Dest) != ".JPG")
  else {
    TJPEGImage        *JPEG = new TJPEGImage;
    Graphics::TBitmap *TempBMP = new Graphics::TBitmap;

    try {
      AnsiString TempFile = TempFileName(".BMP");
      if (!ConvertImage(Source, TempFile))
        throw Exception(String("Error while converting ") + ExtractFileExt(Source) + " to " + ExtractFileExt(Dest) + String(": Could not load source file."));
      TempBMP->LoadFromFile(TempFile);
      JPEG->Assign(TempBMP);
      JPEG->SaveToFile(Dest);
      DeleteFile(TempFile);
    }  // try
    __finally {
      delete TempBMP;
      delete JPEG;
    }  // __finally
  }  // else (ExtractFileExt(Dest) == ".JPG")
  __LEAVEFUNCTION__;
	return true;
}  // bool ConvertImage(AnsiString Source, AnsiString Dest)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// CalcMaxStringWidth()
//
// Calculates the maximum string width in pixels of all the strings in a
// string list based on the given font
//
// Takes:    TStrings pointer to a string list, and a TFont pointer
// Returnns: Width in pixels of the longest string
//---------------------------------------------------------------------------
int CalcMaxStringWidth( TStrings * List, TWinControl *Control)
{
  __ENTERFUNCTION__;
  int MaxWidth = 0;

  TCanvas *Canvas = new TCanvas;
  #ifdef _DEBUG
  Log->LogEntry(__FILE__, __LINE__, "CalcMaxStringWidth: Canvas created");
  #endif
  Canvas->Handle = GetDC(Control->Handle);
  #ifdef _DEBUG
  Log->LogEntry(__FILE__, __LINE__, "CalcMaxStringWidth:  Canvas DC = 0x%4.4X", Canvas->Handle);
  #endif
  // Canvas->Font->Assign(Font);
  for (int n = 0; n < List->Count; n++) {
    int Width = Canvas->TextWidth(List->Strings[n]);
    MaxWidth = Width>MaxWidth?Width:MaxWidth;
  }  // for (int n = 0; n < List->Count; n++)
  // DeleteDC(Canvas->Handle);
  // Canvas->Handle = 0;
  #ifdef _DEBUG
  Log->LogEntry(__FILE__, __LINE__, "CalcMaxStringWidth:  Deleting Canvas");
  #endif
  delete Canvas;

  __LEAVEFUNCTION__;
  return MaxWidth + 5;
}  // int CalcMaxStringWidth( TStrings * List, TFont * Font)
//---------------------------------------------------------------------------
 #pragma package(smart_init)
//---------------------------------------------------------------------------




