//---------------------------------------------------------------------------
// TVMesh.cpp
//
// Self-contained VMesh data (Microsoft Freelancer 3D Model format) handler
// class
//
// Written by Kurt Fitzner <kfitzner@excelcia.org>
// Copyright  2003, Kurt Fitzner
//---------------------------------------------------------------------------
// This file is part of msCMPImporter.
//
// msCMPImporter is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// msCMPImporter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

#include <vcl.h>
#include <windows.h>
#pragma hdrstop

#include "TProgramLog.h"
#include "TVMesh.h"

#pragma package(smart_init)


//---------------------------------------------------------------------------
// TMeshGroups implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TMeshGroups::Add()
//
// Overloaded Add function to support creating and adding a TMeshGroup all in
// one swell foop.
//
// Takes:   All the data of a TMeshGroup
// Returns: Index of added item
// Throws:  bad_alloc if the new fails
//---------------------------------------------------------------------------
int __fastcall TMeshGroups::Add(AnsiString Name, int StartMesh, int MeshCount, int StartVertex, int EndVertex, int StartVertexRef, int EndVertexRef, float OriginX, float OriginY, float OriginZ)
{
  __ENTERFUNCTION__;
  TMeshGroup *MeshGroup = new TMeshGroup;
  MeshGroup->Name           = Name;
  MeshGroup->StartMesh      = StartMesh;
  MeshGroup->MeshCount      = MeshCount;
  MeshGroup->StartVertex    = StartVertex;
  MeshGroup->EndVertex      = EndVertex;
  MeshGroup->StartVertexRef = StartVertexRef;
  MeshGroup->EndVertexRef   = EndVertexRef;
  MeshGroup->OriginX        = OriginX;
  MeshGroup->OriginY        = OriginY;
  MeshGroup->OriginZ        = OriginZ;
  int retval = Add(MeshGroup);
  __LEAVEFUNCTION__;
  return retval;
}  // int __fastcall TMeshGroups::Add(AnsiString Name, int StartMesh, int MeshCount, int StartVertex, int EndVertex, int StartVertexRef, int EndVertexRef, float OriginX, float OriginY, float OriginZ)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TMeshGroups::IndexOf()
//
// Overloaded IndexOf that searches the TList-derived TMeshGroups for a
// TMeshGroup by name.
//
// Takes:   Ansistring name
// Returns: Index of TMeshGroup inside TMeshGroups, or -1 if not found
// Throws:  Nothing
//---------------------------------------------------------------------------
int __fastcall TMeshGroups::IndexOf(AnsiString Name)
{
  __ENTERFUNCTION__;
  for (int n = 0; n < Count; n++)
    if (Items[n]->Name == Name) {
      __LEAVEFUNCTION__;
      return n;
    }  // if (Items[n]->Name == Name)
  __LEAVEFUNCTION__;
  return -1;
}  // int   __fastcall  IndexOf(AnsiString Name)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// End of TMeshGroups implementation
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TMaterials implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TMaterials::IndexOfID()
//
// Perform a binary-tree search for the given MaterialID in the list.
//
// Takes:   unsigned Material ID
// Returns: Index of TMaterial with the given ID number, or -1 if not found
// Throws:  Nothing
//---------------------------------------------------------------------------
int __fastcall TMaterials::IndexOf(DWORD MaterialID)
{
  __ENTERFUNCTION__;
  if (!Count) {
    __LEAVEFUNCTION__;
    return -1;
  }  // if (!Count)

  int Lower = 0;
  int Upper = Count - 1;
  int Index = Upper / 2;

  while (Items[Index]->MaterialID != MaterialID) {
    if (MaterialID > Items[Index]->MaterialID)
      Lower = Index + 1;
    else
      Upper = Index - 1;
    if (Lower > Upper)
      return -1;
    Index = (Upper - Lower) / 2 + Lower;
  }  // while (Items[Index]->MaterialID != MaterialID)
  __LEAVEFUNCTION__;
  return Index;
}  // int __fastcall TMaterials::IndexOfID(DWORD MaterialID)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TMaterials::Add()
//
// Overloaded Add function to support creating and adding a TMaterial all in
// one swell foop.
//
// Takes:   All the data of a TMaterial
// Returns: Index of added item
// Throws:  bad_alloc if the new fails
//---------------------------------------------------------------------------
int __fastcall TMaterials::Add(DWORD MaterialID, AnsiString Name, AnsiString Library, AnsiString FileName, int LibraryNode, bool IsScanned)
{
  __ENTERFUNCTION__;
  TMaterial *Material = new TMaterial;
  Material->MaterialID  = MaterialID;
  Material->Name        = Name;
  Material->Library     = Library;
  Material->FileName    = FileName;
  Material->LibraryNode = LibraryNode;
  Material->IsScanned   = IsScanned;
  int retval = Add(Material);
  __LEAVEFUNCTION__;
  return retval;
}  // int __fastcall TMaterials::Add(DWORD MaterialID, AnsiString Name)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// GetScannedCount()
//
// ScannedCount __property read access method
//---------------------------------------------------------------------------
int __fastcall TMaterials::GetScannedCount(void) {
  __ENTERFUNCTION__;
  int Counter = 0;
  for (int n = 0; n < Count; n++)
    if (Items[n]->IsScanned)
      Counter++;
  __LEAVEFUNCTION__;
  return Counter;
}  // int __fastcall TMaterials::GetScannedCount(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// End of TMaterials implementation
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TMeshes implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TMeshes::Add()
//
// Overloaded Add function to support creating and adding a TMesh all in one
// swell foop.
//
// Takes:   All the data of a TMesh
// Returns: Index of added item
// Throws:  bad_alloc if the new fails
//---------------------------------------------------------------------------
int __fastcall TMeshes::Add(DWORD MaterialID, int StartVertex, int EndVertex, int VertexRefCount)
{
  __ENTERFUNCTION__;
  TMesh *Mesh = new TMesh(this);
  Mesh->MaterialID     = MaterialID;
  Mesh->RelStartVertex = StartVertex;
  Mesh->RelEndVertex   = EndVertex;
  Mesh->VertexRefCount = VertexRefCount;
  int retval = Add(Mesh);
  __LEAVEFUNCTION__;
  return retval;
}  // int __fastcall TMeshes::Add(DWORD MaterialID, int StartVertex, int EndVertex, int VertexRefCount)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// End of TMeshes implementation
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TTriangles implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TTriangles::Add()
//
// Overloaded Add function to support creating and adding a TTriangle all in
// one swell foop.
//
// Takes:   All the data of a TTriangle
// Returns: Index of added item
// Throws:  bad_alloc if the new fails
//---------------------------------------------------------------------------
int __fastcall TTriangles::Add(int Vertex1, int Vertex2, int Vertex3)
{
  __ENTERFUNCTION__;
  TTriangle *Triangle = new TTriangle;
  Triangle->Vertex1 = Vertex1;
  Triangle->Vertex2 = Vertex2;
  Triangle->Vertex3 = Vertex3;
  int retval = Add(Triangle);
  __LEAVEFUNCTION__;
  return retval;
}  // int __fastcall TTriangles::Add(int Vertex1, int Vertex2, int Vertex3)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// End of TTriangles implementation
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TVertices implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TVertices::Add()
//
// Overloaded Add function to support creating and adding a TVertex all in
// one swell foop.
//
// Takes:   All the data of a TVertex
// Returns: Index of added item
// Throws:  bad_alloc if the new fails
//---------------------------------------------------------------------------
int __fastcall TVertices::Add(float X, float Y, float Z, float NormalX, float NormalY, float NormalZ, float U, float V)
{
  __ENTERFUNCTION__;
  TVertex *Vertex = new TVertex;
  Vertex->X = X;
  Vertex->Y = Y;
  Vertex->Z = Z;
  Vertex->NormalX = NormalX;
  Vertex->NormalY = NormalY;
  Vertex->NormalZ = NormalZ;
  Vertex->U = U;
  Vertex->V = V;
  int retval = Add(Vertex);
  __LEAVEFUNCTION__;
  return retval;
}  // int __fastcall TVertices::Add(float X, float Y, float Z, float NormalX, float NormalY, float NormalZ, float U, float V)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// End of TVertices implementation
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TVMesh implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TVMesh::TVMesh()
//
// Default constructor for TVMesh class.  Set up our TList-derived lists.
//
// Takes:   Nothing
// Returns: It's a constructor
// Throws:  bad_alloc if the "new"s fail.
//---------------------------------------------------------------------------
__fastcall TVMesh::TVMesh(void) : TObject()
{
  __ENTERFUNCTION__;
  MeshGroups = new TMeshGroups(this);
  Materials  = new TMaterials(this);
  Meshes     = new TMeshes(this);
  Triangles  = new TTriangles(this);
  Vertices   = new TVertices(this);
  __LEAVEFUNCTION__;
}  // bool TVMesh::TVMesh(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TVMesh::~TVMesh()
//
// Default anticonstructor for TVMesh class.  Set up our TList-derived lists.
//
// Takes:   Nothing
// Returns: It's a destructor
// Throws:  Nothing
//---------------------------------------------------------------------------
__fastcall TVMesh::~TVMesh(void)
{
  __ENTERFUNCTION__;
  delete Vertices;
  delete Triangles;
  delete Meshes;
  delete Materials;
  delete MeshGroups;
  __LEAVEFUNCTION__;
}  // bool TVMesh::TVMesh(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TVMesh::Clear()
//
// Clear out whatever we have loaded (if anyything) so we can start again.
//
// Takes:   Nothing.
// Returns: Nothing.
// Throws:  Nothing.
//---------------------------------------------------------------------------
void __fastcall TVMesh::Clear(void)
{
  __ENTERFUNCTION__;
  Vertices->Clear();
  Triangles->Clear();
  Meshes->Clear();
  Materials->Clear();
  MeshGroups->Clear();
  __LEAVEFUNCTION__;
}  // void __fastcall TVMesh::Clear(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TVMesh::LoadFromStream()
//
// Load the VMesh from an already open stream.  Just in case you're really
// lazy and can't be bothered to rewind the stream, then you can get this
// method to do it for you too.  Wasn't that nice?
//
// Takes:   TStream (or derived class) pointer, and an optional boolean flag
//          to specify if you want the stream rewound first.
// Returns: True on success, false if it doesn't understand the data in
//          the stream.
// Throws:  EReadError if any of the stream reads fail to get enough data.
//---------------------------------------------------------------------------
bool __fastcall TVMesh::LoadFromStream(TStream *InStream, bool Rewind)
{
  __ENTERFUNCTION__;
  TVMeshRawHeader Header;

  if (Rewind)
    __NOEXCEPT(InStream->Seek(0, soFromBeginning));

  InStream->ReadBuffer(&Header, sizeof(TVMeshRawHeader));
  if (Header.Signature1 != 0x00000001 || Header.Signature2 != 0x00000004) {
    __LEAVEFUNCTION__;
    return false;
  }  // if (Header.Signature1 != 0x00000001 || Header.Signature2 != 0x00000004)

  Clear();

  // Load in the Meshes, and create a new material for each unique MaterialID
  // that we find.
  for (int n = 0; n < Header.MeshCount; n++) {
    TVMeshRawMesh RawMesh;
    InStream->ReadBuffer(&RawMesh, sizeof(TVMeshRawMesh));
    Meshes->Add(RawMesh.MaterialID, RawMesh.StartVertex, RawMesh.EndVertex, RawMesh.VertexRefCount);
    if (Materials->IndexOf(RawMesh.MaterialID) == -1)
      Materials->Add(RawMesh.MaterialID, IntToHex((int)RawMesh.MaterialID, 8));
  }  // for (int n = 0; n < Header.MeshCount; n++)

  // Load in the triangles
  for (int n = 0; n < Header.VertexRefCount / 3; n++) {
    TVMeshRawTriangle RawTriangle;
    InStream->ReadBuffer(&RawTriangle, sizeof(TVMeshRawTriangle));
    Triangles->Add(RawTriangle.Vertex1, RawTriangle.Vertex2, RawTriangle.Vertex3);
  }  // for (int n = 0; n < Header.VertexRefCount / 3; n++)

  // Load in the vertices
  for (int n = 0; n < Header.VertexCount; n++) {
    TVMeshRawVertex RawVertex;
    InStream->ReadBuffer(&RawVertex, sizeof(TVMeshRawVertex));
    Vertices->Add(RawVertex.X, RawVertex.Y, RawVertex.Z, RawVertex.NormalX, RawVertex.NormalY, RawVertex.NormalZ, RawVertex.U, RawVertex.V);
  }  // for (int n = 0; n < Header.VertexCount; n++)

  __LEAVEFUNCTION__;
  return true;
}  // bool TVMesh::LoadFromStream(TStream *InStream)
//---------------------------------------------------------------------------




