//---------------------------------------------------------------------------
// TUTFFile.cpp
//
// Self-contained UTF File (Microsoft Freelancer heirarchical database)
// handler class.
//
// Written by Kurt Fitzner <kfitzner@excelcia.org>
// Copyright  2003, Kurt Fitzner
//---------------------------------------------------------------------------
// This file is part of msCMPImporter.
//
// msCMPImporter is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// msCMPImporter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

#include <vcl.h>
#include <windows.h>
#pragma hdrstop

#include <stdio.h>
#include "TProgramLog.h"
#include "TUTFFile.h"

#pragma package(smart_init)


//---------------------------------------------------------------------------
// TUTFFile Class Implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TUTFFile()
//
// Default Constructor
//
// Takes:   Nothing
// Returns: It's a constructor
// Throws:  bad_alloc if it can't make a new TList
//---------------------------------------------------------------------------
__fastcall TUTFFile::TUTFFile(void) : System::TObject()
{
  __ENTERFUNCTION__;
  Nodes = new TList;
  __LEAVEFUNCTION__;
}  // __fastcall TUTFFile::TUTFFile(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ~TUTFFile()
//
// Default destructor
//
// Takes:   Nothing
// Returns: It's a destructor
// Throws:  Nothing but tantrums
//---------------------------------------------------------------------------
__fastcall TUTFFile::~TUTFFile(void)
{
  __ENTERFUNCTION__;
  Clear();
  delete Nodes;
  __LEAVEFUNCTION__;
}  // __fastcall TUTFFile::TUTFFile(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Clear()
//
// Reset the object to nil - free up all the wasted... er used memory
//
// Takes:   Nothing
// Returns: Nothing
// Throws:  Nothing
//---------------------------------------------------------------------------
void __fastcall TUTFFile::Clear(void)
{
  TUTFTreeNode *node;

  __ENTERFUNCTION__;

  if (Nodes)
    for (int n = 0; n < Nodes->Count; n++) {
      node = (TUTFTreeNode *)Nodes->Items[n];
      if (node->Data)
        delete node->Data; // free(node->Data);
        delete node;
    }  // while (Nodes->Count)
    Nodes->Clear();
  __LEAVEFUNCTION__;
}  // void __fastcall TUTFFile::Clear(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// LoadFromFile()
//
// Loads a UTF file into memory from a disk file.
//
// Takes: Ansistring file name
// Returns: bool true if loads ok, false if the file isn't a UTF file.
// Throws: EFOpenError if it can't fopen the file
//         EReadError if it can't read the file
//         bad_alloc if it can't allocate memory
//---------------------------------------------------------------------------
bool __fastcall TUTFFile::LoadFromFile(AnsiString &FileName)
{
  int             UTFFile;
  long int        size, read;
  BYTE           *buffer = NULL;
  TUTFFileHeader *Header;
  int            *Parents = NULL;

  __ENTERFUNCTION__;

  UTFFile = FileOpen(FileName.c_str(), fmOpenRead);
  if (UTFFile == -1)
    throw EFOpenError("Could not open UTF File: '" + String(strerror(errno)) + "'");

  size = FileSeek(UTFFile, 0, 2);
  FileSeek(UTFFile, 0, 0);
  buffer = new BYTE[size]; // (BYTE *) malloc(size);
  if (!buffer)
    OutOfMemoryError();

  try {
    read = FileRead(UTFFile, buffer, size);
    if ( read !=  size)
      throw EReadError("Could not read UTF file: '" + String(strerror(errno)) + "'");
    FileClose(UTFFile);
    Header = (TUTFFileHeader *) buffer;
    if (Header->Signature1 != 0x20465455 || Header->Signature2 != 0x0101) {
      delete buffer;
      __LEAVEFUNCTION__;
      return(false);
    }  // if (Header->Signature1 != 0x20465455 || Header->Signature2 != 0x0101)
    Clear();
    Parents = new int[Header->TreeSegmentSize/44]; //  (int *) calloc(Header->TreeSegmentSize, sizeof(int));
    if (!Parents)
      OutOfMemoryError();

    for (unsigned n = 0; n < Header->TreeSegmentSize / 44; n++) {

      TUTFFileTreeNode *FileNode = (TUTFFileTreeNode *)(buffer + Header->TreeSegmentOffset + n * 44);
      TUTFTreeNode *node = new TUTFTreeNode(this);

      node->Name = String((char *) buffer + Header->StringSegmentOffset + FileNode->StringOffset);
      node->IsLeaf = FileNode->Flags & 0x80;
      node->Child = (node->IsLeaf)?0:FileNode->ChildOffset / 44;
      Parents[node->Child] = n;
      node->Sibling = FileNode->SiblingOffset / 44;
      node->Parent = Parents[n];
      if (node->Parent && node->Sibling) {
        TUTFTreeNode *sibling = node;

        Parents[node->Sibling] = node->Parent;
        // Some malformed UTF files had siblings in the wrong order in the file.
        // The protocol allows it (or seems to), but it's stupid to write it this way.
        while (sibling->Sibling < n) {
          sibling = (TUTFTreeNode *)Nodes->Items[sibling->Sibling];
          sibling->Parent = node->Parent;
          Parents[sibling->Sibling] = n;
        }  // while (sibling->Sibling < n)
      }  // if (node->Parent & node->Sibling)

      // In the case (which will probably never happen) that a node's child
      // was read in before its parent, we check and set the parent attribute
      // on the child.
      if (node->Child && node->Child < n)
        ((TUTFTreeNode *)Nodes->Items[node->Child])->Parent = n;
      node->AllocatedDataSize = FileNode->AllocatedSize;
      node->DataSize = FileNode->Size1;
      if (node->AllocatedDataSize && !node->Child) {
        node->Data = new char[node->AllocatedDataSize]; // GetMem(node->AllocatedDataSize);
        if (!node->Data)
          OutOfMemoryError();
        Move(buffer + Header->DataSegmentOffset + FileNode->ChildOffset, node->Data, node->DataSize);
      } else
        node->Data = NULL;
      node->Unknown = FileNode->Unknown;
      node->Time1   = FileNode->Time1;
      node->Time2   = FileNode->Time2;
      node->Time3   = FileNode->Time3;
      Nodes->Add(node);

    }  // for (int n = 0; n < Header->TreeSegmentSize / 44; n++)

    delete buffer;
    delete Parents;

  }  // try

  catch(Exception &e) {
    if (buffer)
      delete buffer;                         // Free up the mem if we fail
    if (Parents)
      delete Parents;
    while (Nodes->Count > 0) {
      TUTFTreeNode *node = (TUTFTreeNode *)Nodes->Items[0];
      if (node->Data)
        delete node->Data;
      delete node;
      Nodes->Delete(0);
    }  // while (Nodes->Count > 0)
    #ifdef _DEBUG
    Log->LogEntry(__FILE__, __LINE__, "Exception class %s in function %s.", e.ClassName(), __FUNC__);
    #endif
    __LEAVEFUNCTION__;
    throw;
  }  // catch

  __LEAVEFUNCTION__;
  return(true);
}  // bool __fastcall Open(AnsiString &FileName)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// SaveToFile()
//
// Saves a UTF file from memory to a disk file.
//
// Takes: Ansistring file name
// Returns: bool true if saves ok, false if not
// Throws: EFOpenError if it can't fopen the file
//         EWriteError if it can't read the file
//         bad_alloc if it can't allocate memory
//---------------------------------------------------------------------------
bool __fastcall TUTFFile::SaveToFile(AnsiString &FileName)
{
  __ENTERFUNCTION__;

  if (Nodes && Nodes->Count) {
    TUTFFileHeader FileHeader;
    int UTFFile;
    int StringOffset, DataOffset;
    const int Zero = 0;

    try {
      UTFFile = FileCreate(FileName);
      if (UTFFile == -1)
        throw EFOpenError("Could not open file for writing:\r\n" + String(strerror(errno)));
      FileSeek(UTFFile, 0, 0);
      if (!SetEndOfFile((HANDLE)UTFFile))
        throw EWriteError("Could not write to file while truncating:\r\n" + String(strerror(errno)));

      // Calculate sizes and make a header
      FileHeader.Signature1 = 0x20465455;
      FileHeader.Signature2 = 0x0101;
      FileHeader.TreeSegmentOffset = 56;
      FileHeader.TreeSegmentSize = Nodes->Count * sizeof(TUTFFileTreeNode);
      FileHeader.HeaderOffset = 0;
      FileHeader.HeaderSize = 44;
      FileHeader.StringSegmentOffset = FileHeader.TreeSegmentOffset + FileHeader.TreeSegmentSize;
      FileHeader.StringSegmentAllocatedSize = 1;
      for (int n=0; n<Nodes->Count; n++)
        FileHeader.StringSegmentAllocatedSize += ((TUTFTreeNode *)Nodes->Items[n])->Name.Length() + 1;
      FileHeader.StringSegmentSize = FileHeader.StringSegmentAllocatedSize;
      FileHeader.DataSegmentOffset = FileHeader.StringSegmentOffset + FileHeader.StringSegmentSize;
      FileHeader.Unknown = 0;

      // Write the header
      if (FileWrite(UTFFile, &FileHeader, sizeof(TUTFFileHeader)) != sizeof(TUTFFileHeader))
        throw EWriteError("Could not write to file while writing header:\r\n" + String(strerror(errno)));

      // Write twelve bytes of zeros for filler
      FileWrite(UTFFile, &Zero, 4);
      FileWrite(UTFFile, &Zero, 4);
      FileWrite(UTFFile, &Zero, 4);

      // Write TreeSegment
      StringOffset = 1;
      DataOffset = 0;
      for (int n=0; n<Nodes->Count; n++) {
        TUTFFileTreeNode  FileNode;
        TUTFTreeNode     *Node = (TUTFTreeNode *)Nodes->Items[n];

        // Set up the node
        FileNode.SiblingOffset = Node->Sibling * sizeof(TUTFFileTreeNode);
        FileNode.StringOffset = StringOffset;
        FileNode.Flags = Node->IsLeaf?0x80:0x10;
        FileNode.Unknown = Node->Unknown;
        FileNode.ChildOffset = Node->IsLeaf?DataOffset:Node->Child*sizeof(TUTFFileTreeNode);
        FileNode.AllocatedSize = Node->AllocatedDataSize;
        FileNode.Size1 = FileNode.Size2 = Node->DataSize;
        FileNode.Time1 = Node->Time1;
        FileNode.Time2 = Node->Time2;
        FileNode.Time3 = Node->Time3;

        // Write the node
        if (FileWrite(UTFFile, &FileNode, sizeof(TUTFFileTreeNode)) != sizeof(TUTFFileTreeNode))
          throw EWriteError("Could not write to file while writing node:\r\n" + String(strerror(errno)));

        // Increment segment offsets
        StringOffset += Node->Name.Length() + 1;
        DataOffset += Node->AllocatedDataSize;
      }  // for (int n=0; n<Nodes->Count; n++)

      // Write the string segment
      // String segments always have a leading null byte - dunno why
      FileWrite(UTFFile, &Zero, 1);
      for (int n=0; n<Nodes->Count; n++) {
        TUTFTreeNode     *Node = (TUTFTreeNode *)Nodes->Items[n];

        FileWrite(UTFFile, Node->Name.c_str(), Node->Name.Length());
        FileWrite(UTFFile, &Zero, 1);
      }  // for (int n=0; n<Nodes->Count; n++)

      // Write the data segment
      for (int n=0; n<Nodes->Count; n++) {
        TUTFTreeNode     *Node = (TUTFTreeNode *)Nodes->Items[n];

        if (Node->IsLeaf && Node->AllocatedDataSize)
          FileWrite(UTFFile, Node->Data, Node->AllocatedDataSize);
      }  // for (int n=0; n<Nodes->Count; n++)

      // FileClose(UTFFile);

    }  // try
    __finally {
      if (UTFFile != -1)
        FileClose(UTFFile);
      #ifdef _DEBUG
      Log->LogEntry(__FILE__, __LINE__, "__finally executed in function %s.", __FUNC__);
      #endif
      __LEAVEFUNCTION__;
    }  // __finally
    __LEAVEFUNCTION__;
    return true;
  } else {
    __LEAVEFUNCTION__;
    return false;
  }  // if (Nodes && Nodes->Count)


}  // bool __fastcall TUTFFile::SaveToFile(AnsiString &FileName)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// SetData(int Index, AnsiString NewData)
//
// Set a UTF node's data to be the given string, freeing previous data
// if any.
//
// Takes: Integer index indicating which node, and an AnsiString containing
//        the data
// Returns: Nothing
// Throws: EOutOfMemory if it can't alloc enough space
//---------------------------------------------------------------------------
void __fastcall TUTFFile::SetData(int Index, AnsiString NewData)
{
  TUTFTreeNode *Node;
  void         *Data;
  int           DataSize;
  const int     Length = NewData.Length() + 1;

  __ENTERFUNCTION__;

  if (Index > Nodes->Count)
    return;

  DataSize = (Length%4)?(Length/4+1)*4:Length;
  Data = calloc(DataSize, 1);
  if (!Data)
    OutOfMemoryError();
  memcpy(Data, NewData.c_str(), Length);
  Node = (TUTFTreeNode *)Nodes->Items[Index];
  if (Node->Data)
    free(Node->Data);
  Node->Data = Data;

  __LEAVEFUNCTION__;
}  // void __fastcall TUTFFile::SetData(int Index, AnsiString NewData)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// IndexOf()
//
// Find a node by name
//
// Takes:   Ansistring node name.  If the name is prefixed by a backslash,
//          then the name is assumed to be a node path.
// Returns: Index of node in list, or -1 if not found
// Throws:  Nothing
//---------------------------------------------------------------------------
int  __fastcall TUTFFile::IndexOf(AnsiString Name)
{
  __ENTERFUNCTION__;

  if (Name.SubString(0, 1) != "\\") {
    for (int n=0; n<Count; n++)
      if (!CompareText(Items[n]->Name,Name)) {
        __LEAVEFUNCTION__;
        return n;
      }  // if (!CompareText(Items[n]->Name,Name))
    __LEAVEFUNCTION__;
    return -1;
  }  // if (Name.SubString(0, 1) != "\\")
  else {
    TStringList *Path = new TStringList;
    unsigned Index = 0;

    ExtractStrings(TSysCharSet() << '\\', TSysCharSet(), Name.c_str(), Path);
    for (int n = 0; n < Path->Count; n++)
      if ((Index = Items[Index]->ChildByName(Path->Strings[n])) == -1)
        break;
    delete Path;
    __LEAVEFUNCTION__;
    return Index;
  }  // else (Name.SubString(0, 1) == "\\")
}  // int  __fastcall TUTFFile::IndexOf(AnsiString Name)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// GetCount()
//
// Read access method for TUTFFile::Count property
//---------------------------------------------------------------------------
int __fastcall TUTFFile::GetCount(void)
{
  __ENTERFUNCTION__;

  if (Nodes) {
    __LEAVEFUNCTION__;
    return(Nodes->Count);
  } else {
    __LEAVEFUNCTION__;
    return(0);
  }  // if (Nodes)
}  // int __fastcall TUTFFile::GetCount(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// GetItems()
//
// Read access method for TUTFFile::Items[] property
//---------------------------------------------------------------------------
TUTFTreeNode * __fastcall TUTFFile::GetItems(int Index)
{
  __ENTERFUNCTION__;
  if (Nodes) {
    __LEAVEFUNCTION__;
    return((TUTFTreeNode *)Nodes->Items[Index]);
  } else {
    __LEAVEFUNCTION__;
    return(NULL);
  }  // if (Nodes)
}  // int __fastcall TUTFFile::GetItems(int Index)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// GetNames()
//
// Read access method for TUTFFile::Names[] property
//---------------------------------------------------------------------------
AnsiString __fastcall TUTFFile::GetNames(int Index)
{
  __ENTERFUNCTION__;
  if (Nodes) {
    __LEAVEFUNCTION__;
    return(((TUTFTreeNode *)Nodes->Items[Index])->Name);
  } else {
    __LEAVEFUNCTION__;
    return("");
  }  // if (Nodes)
}  // int __fastcall TUTFFile::GetNames(int Index)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// PutNames()
//
// Write access method for TUTFFile::Names[] property
//---------------------------------------------------------------------------
void __fastcall TUTFFile::PutNames(int Index, AnsiString Name)
{
  __ENTERFUNCTION__;
  if (Nodes)
    ((TUTFTreeNode *)Nodes->Items[Index])->Name = Name;
  __LEAVEFUNCTION__;
}  // int __fastcall TUTFFile::PutNames(AnsiString Name, int Index)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TUTFTreeNode Class Implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TUTFTreeNode()
//
// Default constructor
//
// Takes:   Nothing
// Returns: It's a constructor
// Throws:  Nothing
//---------------------------------------------------------------------------
__fastcall TUTFTreeNode::TUTFTreeNode(TUTFFile *MyOwner)
{
  __ENTERFUNCTION__;

  Name = "";
  Child = Sibling = Parent = 0;
  IsLeaf = false;
  AllocatedDataSize = DataSize = 0;
  Data = NULL;
  Unknown = Time1 = Time2 = Time3 = 0;
  Owner = MyOwner;

  __LEAVEFUNCTION__;
}  // __fastcall TUTFTreeNode::TUTFTreeNode(void)


//---------------------------------------------------------------------------
// ~TUTFTreeNode()
//
// Default destructor
//
// Takes:   Nothing
// Returns: It's a destructor
// Throws:  Nothing
//---------------------------------------------------------------------------
__fastcall TUTFTreeNode::~TUTFTreeNode()
{
  __ENTERFUNCTION__;
  __LEAVEFUNCTION__;
}  // __fastcall TUTFTreeNode::~TUTFTreeNode()


//---------------------------------------------------------------------------
// GetGrandParent()
//
// Access method for the GrandParent property
//---------------------------------------------------------------------------
unsigned __fastcall TUTFTreeNode::GetGrandParent(void)
{
  __ENTERFUNCTION__;

  if (!Parent) {
    __LEAVEFUNCTION__;
    return 0;
  }  // (if !Parent)
  __LEAVEFUNCTION__;
  return ((TUTFTreeNode *)Owner->Nodes->Items[Parent])->Parent;
}  // int __fastcall GetGrandParent(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// ChildByName()
//
// Find a child of a specific node by name.  Checks only immediate children,
// not grandchildren etc.
//
// Takes:   Ansistring name of child node to look for
// Returns: Index of the child node, or -1 if it doesn't exist
// Throws:  Nothing
//---------------------------------------------------------------------------
unsigned __fastcall TUTFTreeNode::ChildByName(AnsiString Name)
{
  unsigned Index;

  __ENTERFUNCTION__;

  if (!Child)
    return -1;

  Index = Child;
  while (Index)
    if (!CompareText(Owner->Items[Index]->Name,Name)) {
      __LEAVEFUNCTION__;
      return Index;
    } else
      Index = Owner->Items[Index]->Sibling;
  __LEAVEFUNCTION__;
  return -1;
}  // unsigned __fastcall TUTFTreeNode::ChildByName(AnsiString Name)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// SaveToFile()
//
// Save a node's data to a file
//
// Takes:   AnsiString filename
// Returns: true on success, falst on fail or if there is no data to save
// Throws:  Dunno yet
//---------------------------------------------------------------------------
bool __fastcall TUTFTreeNode::SaveToFile(AnsiString FileName)
{
  __ENTERFUNCTION__;

  if (DataSize > 0) {
    TFileStream *SaveStream = new TFileStream(FileName,fmCreate);
    SaveStream->Size = 0;
    SaveStream->Seek(0,soFromBeginning);
    SaveStream->Write(Data, DataSize);
    delete SaveStream;
    __LEAVEFUNCTION__;
    return true;
  }  // if (DataSize > 0)
  else {
    __LEAVEFUNCTION__;
    return false;
  }
}  // bool __fastcall SaveToFile(AnsiString FileName)
//---------------------------------------------------------------------------


