//---------------------------------------------------------------------------
// msCMPImporter.dll - TCMPImporter.cpp
//
// Main import form and code.
//
// Written by Kurt Fitzner <kfitzner@excelcia.org>
// Copyright  2003, Kurt Fitzner
//---------------------------------------------------------------------------
// This file is part of msCMPImporter.
//
// msCMPImporter is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// msCMPImporter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------

#include <vcl.h>
#include <windows.h>
#pragma hdrstop

#include <FileCtrl.hpp>
#include <Registry.hpp>
#include <shlobj.h>

#include "TUTFFile.h"
#include "TVMesh.h"
#include "Utilities.h"
#include "TProgramLog.h"
#include "MSLib.h"
#include "TCMPImporter.h"

#pragma package(smart_init)
#pragma link "CSPIN"
#pragma resource "*.dfm"


//---------------------------------------------------------------------------
// TCMPImporter class implementation
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TCMPImporter::TCMPImporter()
//
// Constructor for the form
//---------------------------------------------------------------------------
__fastcall TCMPImporter::TCMPImporter(TComponent* Owner) : TForm(Owner)
{
  __ENTERFUNCTION__;
  CMPFile = NULL;
  VMesh   = NULL;
  CMPFile = new TUTFFile;
  VMesh   = new TVMesh;
  TextureDir = "";
  FirstActivate = true;
  __LEAVEFUNCTION__;
}  // __fastcall TCMPImporter::TCMPImporter(TComponent* Owner)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::~TCMPImporter()
//
// Anti-Constructor for the form
//---------------------------------------------------------------------------
__fastcall TCMPImporter::~TCMPImporter()
{
  __ENTERFUNCTION__;
  if (VMesh)
    delete VMesh;
  if (CMPFile)
    delete CMPFile;
  __LEAVEFUNCTION__;
}  // __fastcall ~TCMPImporter()
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TCMPImporter::FormShow()
//
// Form OnShow event handler.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::FormShow(TObject *Sender)
{
  __ENTERFUNCTION__;
  __LEAVEFUNCTION__;
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// TCMPImporter::FormActivate()
//
// OnActivate event handler.  OnActivate is the last event to fire in a
// form creation event chain (OnCreate, OnShow, OnActivate).  It happens
// just before a form is painted.  Since all the controls are created at
// this point, and the form isn't yet painted to the screen, it's a great
// place to put initialization code.  That way you don't have to worry about
// what state a control is in.  However, OnActivate may happen more than
// once if a form goue out of then back in focus, so we add a static variable
// to test if this is the first time the event is called, and if so, we
// trigger our own FormActivateFirst() "pseudo-event".
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::FormActivate(TObject *Sender)
{
  __ENTERFUNCTION__;
  if (FirstActivate) {
    FormActivateFirst(Sender);
    FirstActivate = false;
  }  // if (First)
  __LEAVEFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::FormActivateFirst()
//
// Additional "pseudo-event"  that is called the first time the form is
// activated.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::FormActivateFirst(TObject *Sender)
{
  __ENTERFUNCTION__;

  edtFileNameOldMessageHandler = edtFileName->WindowProc;
  edtFileName->WindowProc = edtFileNameMessageHandler;
  pnlMaterialsOldMessageHandler = pnlMaterials->WindowProc;
  pnlMaterials->WindowProc = pnlMaterialsMessageHandler;

  TRegistry *Registry;
  Registry = new TRegistry();

  Registry->RootKey = HKEY_LOCAL_MACHINE;
  if (Registry->OpenKey("SOFTWARE\\Microsoft\\Microsoft Games\\Freelancer\\1.0",false))
    FLDir = Registry->ReadString("AppPath");
  delete Registry;
  if (!FLDir.IsEmpty())
    OpenDialog->InitialDir = FLDir + "\\Data";
  __LEAVEFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::Execute()
//
// Main external routine for this class.  This is what others call to
// run this importer dialog
//
// Takes:   Nothing
// Returns: Integer
// Throws:  Nothing
//---------------------------------------------------------------------------
int __fastcall TCMPImporter::Execute(msModel *pModel)
{
  __ENTERFUNCTION__;
  DragAcceptFiles(edtFileName->Handle, true);
  UpdatePanels();
  Model = pModel;
  int retval = ShowModal();
  __LEAVEFUNCTION__;
  return retval;
}  // int __fastcall Execute(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::btnOpenClick(TObject *Sender)
//
// OnClick handler for the file open button
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::btnOpenClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  if (OpenDialog->Execute())
    OpenCMPFile(OpenDialog->FileName);
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::btnOpenClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::edtFileNameMessageHandler()
//
// Custom message handler for the edtFilaName control on the form.  This
// message handler knows how to deal with the WM_DROPFILES windows message
// so that a user can drop a file from Explorer onto the filename control.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::edtFileNameMessageHandler(TMessage &Message)
{
  if (Message.Msg == WM_DROPFILES) {
    __ENTERFUNCTION__;
    TWMDropFiles *DropMessage = (TWMDropFiles *)&Message;
    char buffer[256];

    if (DragQueryFile((void *)DropMessage->Drop,0, buffer, 256))
      OpenCMPFile(buffer);
    __LEAVEFUNCTION__;
  } else
    edtFileNameOldMessageHandler(Message);
}  // void __fastcall TCMPImporter::edtFileNameMessageHandler(TMessage &Message)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//  TCMPImporter::OpenCMPFile()
//
// Try to open the CMP model.  Will load and store the UTF file and then
// go one to parse the LODs it contains.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::OpenCMPFile(AnsiString FileName)
{
  __ENTERFUNCTION__;
  try {
    if (FileExists(FileName)) {
      if (!CMPFile->LoadFromFile(FileName))
        throw EFOpenError("File does not appear to be a Freelancer CMP model in UTF format.");

      ModelFileName = FileName;

      TCanvas *FileNameCanvas = new TCanvas;
      HWND notUsed;
      FileNameCanvas->Handle = GetDeviceContext(notUsed);
      FileNameCanvas->Font->Assign(edtFileName->Font);
      edtFileName->Text = MinimizeName(FileName, FileNameCanvas, edtFileName->Width);
      delete FileNameCanvas;

      CMPFileName = OpenDialog->FileName;
      lbLOD->Items->Clear();
      lbLOD->ItemIndex = -1;
      for (int n=0; n<CMPFile->Count; n++) {
        TUTFTreeNode *node = CMPFile->Items[n];
        if (!CompareText(node->Name,"VMeshData"))
          lbLOD->Items->AddObject(CMPFile->Items[node->Parent]->Name,(TObject *)n);
      }  // for (int n=0; n<CMPFile->Count; n++)
      if (!lbLOD->Items->Count)
        throw Exception("Freelancer UTF file contains no VMeshData nodes - no models present in file");
      else if (lbLOD->Items->Count == 1) {
        lbLOD->ItemIndex = 0;
        lbLODClick(NULL);
      }
    }  // if (OpenDialog->Execute() && FileExists(OpenDialog->FileName))
  }  // try
  catch (Exception &e) {
    MessageDlg(String(e.ClassName()) + " - " + String("Could not open model:\n") + e.Message, mtError, TMsgDlgButtons() << mbOK, 0);
    LOG(__FILE__, __LINE__, "Exception class " + e.ClassName() + " in function " + __FUNC__);
    __LEAVEFUNCTION__;
  }  // catch
  UpdatePanels();
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::OpenCMPFile(AnsiString FileName)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::lbLODClick()
//
// lbLOD listbox OnClick event handler
// Load and parse the VMesh LOD selected by the user, and display the
// materials in that VMesh.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::lbLODClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  if (lbLOD->ItemIndex != -1) {
    int            CMPNodeNum = (int)lbLOD->Items->Objects[lbLOD->ItemIndex];
    TUTFTreeNode  *VMeshNode = CMPFile->Items[CMPNodeNum];
    TBufferStream *VMeshStream;
    unsigned       VMeshID = generate_crc(lbLOD->Items->Strings[lbLOD->ItemIndex].c_str());

    VMesh->Clear();
    lbMaterials->Clear();

    VMeshStream = new TBufferStream(VMeshNode->Data, VMeshNode->AllocatedDataSize);
    VMesh->LoadFromStream(VMeshStream);
    delete VMeshStream;

    for (int n = 0; n < VMesh->Materials->Count; n++) {
      TMaterial *Material = VMesh->Materials->Items[n];
      lbMaterials->Items->AddObject(Material->Name, (TObject *)n);
    }  // for (int n = 0; n < VMesh->Materials->Count; n++)

    int CmpndIndex = CMPFile->IndexOf("\\Cmpnd");

    if (CmpndIndex != -1) {
      TUTFTreeNode  *Cmpnd = CMPFile->Items[CmpndIndex];
      int CmpndSiblingIndex = Cmpnd->Child;
      AnsiString VMeshGroupName;

      // Check out each VMeshRef node - it tells us what meshes are in each "mesh group".
      while (CmpndSiblingIndex) {
        int ObjectNameIndex, FileNameIndex;
        ObjectNameIndex = CMPFile->Items[CmpndSiblingIndex]->ChildByName("Object name");
        FileNameIndex   = CMPFile->Items[CmpndSiblingIndex]->ChildByName("File name");
        if (ObjectNameIndex != -1 && FileNameIndex != -1) {
          AnsiString    VMeshRefParentName = String("\\") + String((char *)CMPFile->Items[FileNameIndex]->Data) + String("\\MultiLevel");
          AnsiString    VMeshRefName;
          int           VMeshRefParentIndex = CMPFile->IndexOf(VMeshRefParentName);
          int           VMeshRefIndex;

          if (VMeshRefParentIndex == -1) {
            CmpndSiblingIndex = CMPFile->Items[CmpndSiblingIndex]->Sibling;
            continue;
          }  // if (VMeshRefParentIndex == -1)
          VMeshRefIndex = CMPFile->Items[VMeshRefParentIndex]->Child;
          while (VMeshRefIndex) {
            int Index = CMPFile->IndexOf(VMeshRefParentName + "\\" + CMPFile->Items[VMeshRefIndex]->Name + "\\VMeshPart\\VMeshRef");
            if (Index != -1 && CMPFile->Items[Index]->AllocatedDataSize >= 60) {
              TRawVMeshRef *VMeshRef = (TRawVMeshRef *)CMPFile->Items[Index]->Data;
              if (VMeshRef->Signature1 == 0x0000003c) {
                if (VMeshRef->VMeshID == VMeshID) {
                  VMesh->MeshGroups->Add((char *)CMPFile->Items[ObjectNameIndex]->Data, VMeshRef->StartMesh, VMeshRef->MeshCount,
                                     VMeshRef->StartVertex, VMeshRef->EndVertex, VMeshRef->StartVertexRef, VMeshRef->EndVertexRef);
                  break;
                }  // if (VMeshRef->VMeshID == VMeshID)
              }  // if (VMeshRef->Signature1 == 0x0000003c)
              else
                throw Exception("Error:  VMeshRef data has wrong signature.  The model .cmp file\r\nmay have corrupted data in it.");
            }  // if (Index != -1 && CMPFile->Items[Index]->AllocatedDataSize >= 60)
            VMeshRefIndex = CMPFile->Items[VMeshRefIndex]->Sibling;
          }  // while (VMeshRefIndex)
        }  // if (ObjectNameIndex != -1 && FileNameIndex != -1)
        CmpndSiblingIndex = CMPFile->Items[CmpndSiblingIndex]->Sibling;
      }  // while (CmpndSiblingIndex)

      // Look in the \Cmpnd\Cons\* nodes - they contain data that links
      // each mesh to the "root" mesh - tells us where the origin of
      // a mesh group is.
      int ConsIndex = Cmpnd->ChildByName("Cons");
      if (ConsIndex != -1) {
        int ConsChildIndex = CMPFile->Items[ConsIndex]->Child;
        while (ConsChildIndex) {
          if (!(CMPFile->Items[ConsChildIndex]->DataSize % sizeof(TRawFixedCons))) {
            for (unsigned n = 0; n < CMPFile->Items[ConsChildIndex]->DataSize / sizeof(TRawFixedCons); n++) {
              TRawFixedCons *Construct = (TRawFixedCons *)CMPFile->Items[ConsChildIndex]->Data + n;
              int MeshGroupIndex = VMesh->MeshGroups->IndexOf(Construct->PartName);
              if (MeshGroupIndex != -1) {
                TMeshGroup *MeshGroup = VMesh->MeshGroups->Items[MeshGroupIndex];
                MeshGroup->OriginX = Construct->OriginX;
                MeshGroup->OriginY = Construct->OriginY;
                MeshGroup->OriginZ = Construct->OriginZ;
              }  // if (MeshGroupIndex != -1)
            }  // for (unsigned n = 0; n < CMPFile->Items[Index]->DataSize / sizeof(TRawConstruct); n++)
          }  // if (!(CMPFile->Items[ConsChildIndex]->DataSize % sizeof(TRawFixedCons)))
          else if (!(CMPFile->Items[ConsChildIndex]->DataSize % sizeof(TRawRevoluteCons))) {
            for (unsigned n = 0; n < CMPFile->Items[ConsChildIndex]->DataSize / sizeof(TRawRevoluteCons); n++) {
              TRawRevoluteCons *Construct = (TRawRevoluteCons *)CMPFile->Items[ConsChildIndex]->Data + n;
              int MeshGroupIndex = VMesh->MeshGroups->IndexOf(Construct->PartName);
              if (MeshGroupIndex != -1) {
                TMeshGroup *MeshGroup = VMesh->MeshGroups->Items[MeshGroupIndex];
                MeshGroup->OriginX = Construct->OriginX;
                MeshGroup->OriginY = Construct->OriginY;
                MeshGroup->OriginZ = Construct->OriginZ;
              }  // if (MeshGroupIndex != -1)
            }  // for (unsigned n = 0; n < CMPFile->Items[Index]->DataSize / sizeof(TRawConstruct); n++)
          }  // else if (!(CMPFile->Items[ConsChildIndex]->DataSize % sizeof(TRawRevoluteCons)))
          else {
            for (unsigned n = 0; n < CMPFile->Items[ConsChildIndex]->DataSize / sizeof(TRawSphereCons); n++) {
              TRawSphereCons *Construct = (TRawSphereCons *)CMPFile->Items[ConsChildIndex]->Data + n;
              int MeshGroupIndex = VMesh->MeshGroups->IndexOf(Construct->PartName);
              if (MeshGroupIndex != -1) {
                TMeshGroup *MeshGroup = VMesh->MeshGroups->Items[MeshGroupIndex];
                MeshGroup->OriginX = Construct->OriginX;
                MeshGroup->OriginY = Construct->OriginY;
                MeshGroup->OriginZ = Construct->OriginZ;
              }  // if (MeshGroupIndex != -1)
            }  // for (unsigned n = 0; n < CMPFile->Items[Index]->DataSize / sizeof(TRawConstruct); n++)
          }  // else
          ConsChildIndex = CMPFile->Items[ConsChildIndex]->Sibling;
        }  // while (ConsChildIndex)
      }  // if (ConsIndex != -1)
    }  // if (CmpndIndex != -1)
  }  // if (lbLOD->ItemIndex != -1)
  ScanMaterials(ModelFileName);
  UpdatePanels();
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::lbLODClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::pnlMaterialsMessageHandler()
//
// Custom message handler for the pnlMaterials panel on the form.  This
// message handler knows how to deal with the WM_DROPFILES windows message
// so that a user can drop files from Explorer onto it and have the importer
// scan the dropped files for materials.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::pnlMaterialsMessageHandler(TMessage &Message)
{
  if (Message.Msg == WM_DROPFILES) {
    __ENTERFUNCTION__;
    TWMDropFiles *DropMessage = (TWMDropFiles *)&Message;
    char buffer[256];
    int DropCount = DragQueryFile((void *)DropMessage->Drop, 0xFFFFFFFF, NULL, 0);

    for (int n = 0; n < DropCount; n++)
      if (DragQueryFile((void *)DropMessage->Drop,n, buffer, 256))
        ScanMaterials(buffer);
    __LEAVEFUNCTION__;
  } else
    pnlMaterialsOldMessageHandler(Message);
}  // void __fastcall TCMPImporter::edtFileNameMessageHandler(TMessage &Message)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// btnScanClick()
//
// btnScan button OnClick event handler
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::btnScanClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  if (ScanDialog->Execute() == mrOk)
    for (int n = 0; n< ScanDialog->Files->Count; n++)
      ScanMaterials(ScanDialog->Files->Strings[n]);
  __LEAVEFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// lbMaterialsDrawItem()
//
// lbMaterials listbox OnDrawItem (owner draw) event handler.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::lbMaterialsDrawItem(TWinControl *Control, int Index, TRect &Rect, TOwnerDrawState State)
{
  __ENTERFUNCTION__;

  TCanvas *Canvas = lbMaterials->Canvas;
  TMaterial *Material = VMesh->Materials->Items[(int)lbMaterials->Items->Objects[Index]];

  Canvas->FillRect(Rect);
  ImageList->Draw(Canvas, Rect.Left, Rect.Top-1, Material->IsScanned, true);
  Canvas->TextOut(Rect.Left + ImageList->Width + 2, Rect.top, lbMaterials->Items->Strings[Index]);

  __LEAVEFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// ScanMaterials()
//
// Scan a Freelancer Material Library file and look for any materials that
// match the 32 bit CRC MaterialID used in any of the materials for the
// currently loaded model.  If we find a match, we store the filename and
// the UTF node number of the material inside the appropriate TMaterial
// object.
//
// Takes:   AnsiString filename of the material library to scan.
// Returns: Nothing
// Throws:  Nothing 
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::ScanMaterials(AnsiString FileName)
{
  __ENTERFUNCTION__;

  TUTFFile *MaterialLibrary = new TUTFFile;
  int MaterialParent;

  try {
    if (!MaterialLibrary->LoadFromFile(FileName))
      return;

    TCanvas *FileNameCanvas = new TCanvas;
    HWND notUsed;
    FileNameCanvas->Handle = GetDeviceContext(notUsed);
    FileNameCanvas->Font->Assign(edtMaterialLibrary->Font);
    edtMaterialLibrary->Text = MinimizeName(FileName, FileNameCanvas, edtMaterialLibrary->Width);
    delete FileNameCanvas;

    MaterialParent = MaterialLibrary->IndexOf("\\material library");
    if (MaterialParent != -1) {
      int LibraryIndex = MaterialLibrary->Items[MaterialParent]->Child;
      while (LibraryIndex) {
        int MaterialID = generate_crc(MaterialLibrary->Items[LibraryIndex]->Name.c_str());
        int MaterialIndex = VMesh->Materials->IndexOf(MaterialID);
        if (MaterialIndex != -1 && !VMesh->Materials->Items[MaterialIndex]->IsScanned) {
          // We've got a match
          int TextureNameNode = MaterialLibrary->Items[LibraryIndex]->ChildByName("Dt_name");
          TMaterial *Material = VMesh->Materials->Items[MaterialIndex];
          Material->Name = MaterialLibrary->Items[LibraryIndex]->Name;
          Material->Library = FileName;
          Material->LibraryNode = LibraryIndex;
          Material->IsScanned = true;
          if (TextureNameNode != -1) {
            int TextureIndex = MaterialLibrary->IndexOf(String("\\texture library\\") + String((char *)MaterialLibrary->Items[TextureNameNode]->Data));
            Material->FileName = MaterialLibrary->Items[TextureIndex]->Name;
          }  // if (TextureNameNode != -1)
          lbMaterials->Items->Strings[MaterialIndex] = Material->Name;
        }  // if (MaterialIndex != -1)
        LibraryIndex = MaterialLibrary->Items[LibraryIndex]->Sibling;
      }  // while (MaterialIndex)

    }  // if (MaterialParent != -1)
    UpdatePanels();
  }  // try
  catch (...) {
    __LEAVEFUNCTION__;
    return;
  }  // catch
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::ScanMaterials(AnsiString FileName)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::UpdatePanels()
//
// Update the status of the three window sections.  Check to see which
// "stage" the user is at and disable further stages, and highlight
// the current one.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::UpdatePanels(void)
{
  __ENTERFUNCTION__;
  if (!CMPFile->Count || !lbLOD->Items->Count)
    UpdatePanel(panelFile, statusCurrent);
  else if (!VMesh->IsLoaded())
    UpdatePanel(panelLOD, statusCurrent);
  else if (VMesh->Materials->ScannedCount < VMesh->Materials->Count)
    UpdatePanel(panelMaterials, statusCurrent);
  else {
    UpdatePanel(panelFile, statusEnabled);
    UpdatePanel(panelLOD, statusEnabled);
    UpdatePanel(panelMaterials, statusEnabled);
  }
  lbLOD->ScrollWidth = CalcMaxStringWidth(lbLOD->Items, lbLOD);
  StatusBar->Panels->Items[0]->Text = String(VMesh->Materials->ScannedCount) + String("/") + String(VMesh->Materials->Count) + String(" materials");
  StatusBar->Panels->Items[1]->Text = String(VMesh->Meshes->Count) + String(" meshes");
  StatusBar->Panels->Items[2]->Text = String(VMesh->Triangles->Count) + String(" triangles");
  StatusBar->Panels->Items[3]->Text = String(VMesh->Vertices->Count) + String(" vertices");
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::UpdatePanels(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// StarusBarDrawPanel()
//
// StatusBar OnDrawPanel event handler
// The status bar panels are set to owner-draw so we can change the color
// to reflect whether the model is OK or not.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::StatusBarDrawPanel(TStatusBar *StatusBar, TStatusPanel *Panel, const TRect &Rect)
{
  __ENTERFUNCTION__;

  TCanvas *Canvas = StatusBar->Canvas;
  TBrush  *Brush  = Canvas->Brush;

  if (Panel == StatusBar->Panels->Items[0]) {
    if (VMesh->Materials->Count > 128) {
      Brush->Style = TBrushStyle() << bsSolid;
      Brush->Color = clRed;
    }  // if (VMesh->Meshes->Count > 128)
    else if (VMesh->Materials->ScannedCount < VMesh->Materials->Count) {
      Brush->Style = TBrushStyle() << bsSolid;
      Brush->Color = clYellow;
    }  // if (VMesh->Materials->ScannedCount)
    else {
      Brush->Style = bsClear;
    }  // else (VMesh->Meshes->Count <= 128)
  }  // if (Panel == StatusBar->Panels->Items[0])
  else if (Panel == StatusBar->Panels->Items[1]) {
    if (VMesh->Meshes->Count > 128) {
      Brush->Style = TBrushStyle() << bsSolid;
      Brush->Color = clRed;
    }  // if (VMesh->Meshes->Count > 128)
    else {
      Brush->Style = bsClear;
    }  // else (VMesh->Meshes->Count <= 128)
  }  // if (Panel == StatusBar->Panels->Items[0])
  else if (Panel == StatusBar->Panels->Items[2]) {
    if (VMesh->Triangles->Count > 16384) {
      Brush->Color = clRed;
      Brush->Style = TBrushStyle() << bsSolid;
    }  // if (VMesh->Meshes->Count > 128)
    else {
      Brush->Style = bsClear;
    }  // else (VMesh->Meshes->Count <= 128)
  }  // if (Panel == StatusBar->Panels->Items[0])
  else if (Panel == StatusBar->Panels->Items[3]) {
    if (VMesh->Vertices->Count > 8192) {
      Brush->Color = clRed;
      Brush->Style = TBrushStyle() << bsSolid;
    }  // if (VMesh->Meshes->Count > 128)
    else {
      Brush->Style = bsClear;
    }  // else (VMesh->Meshes->Count <= 128)
  }  // if (Panel == StatusBar->Panels->Items[0])
  Canvas->FillRect(Rect);
  Canvas->TextRect(Rect, Rect.Left + 2, Rect.Top, Panel->Text);
  __LEAVEFUNCTION__;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// TCMPImporter::UpdatePanel()
//
// Do the actual work of updating a panel.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::UpdatePanel(TPanelIdent Panel, TPanelStatus Status)
{
  __ENTERFUNCTION__;

  switch (Panel) {
    case panelFile:
      if (Status == statusDisabled) {
        throw Exception("Program logic error: Attempt to set panelFile (Step 1 panel) to disabled.  Wierd.");
      }  // if (Status == statusDisabled)
      else {
        if (Status == statusCurrent) {
          pnlFile->Font->Style = TFontStyles() << fsBold;
          // If this panel is current, then the other two are disabled.
          // Isn't recursion grand?  To understant recursion, though,
          // you must first understand recursion.
          UpdatePanel(panelLOD, statusDisabled);
          UpdatePanel(panelMaterials, statusDisabled);
        }  // if (Status == statusCurrent)
        else {
          pnlFile->Font->Style = TFontStyles();
        }  // else (Status != statusCurrent)
      }  // else (Status != statusDisabled)
      break;

    case panelLOD:
      if (Status == statusDisabled) {
        pnlLOD->Font->Style = TFontStyles();
        lblLODPanelCaptionShadow->Visible = true;
        lblLODPanelCaptionHighlight->Visible = true;
        lbLOD->BevelKind = bkFlat;
        lbLOD->Enabled = false;
        //pnlFile->Font->Color =
      }  // if (Status == statusDisabled)
      else {
        lblLODPanelCaptionShadow->Visible = false;
        lblLODPanelCaptionHighlight->Visible = false;
        lbLOD->BevelKind = bkSoft;
        lbLOD->Enabled = true;
        if (Status == statusCurrent) {
          lbLOD->SetFocus();
          pnlLOD->Font->Style = TFontStyles() << fsBold;
          // If this panel is current, then the first is enabled and
          // the last is disabled.
          UpdatePanel(panelFile, statusEnabled);
          UpdatePanel(panelMaterials, statusDisabled);
        }  // if (Status == statusCurrent)
        else {
          pnlLOD->Font->Style = TFontStyles();
        }  // else (Status != StatusCurrent)
      }  // else (Status != statusDisabled)
      break;

    case panelMaterials:
      if (Status == statusDisabled) {
        pnlMaterials->Font->Style = TFontStyles();
        lblMaterialPanelCaptionShadow->Visible = true;
        lblMaterialPanelCaptionHighlight->Visible = true;
        edtMaterialLibrary->BevelKind = bkFlat;
        edtMaterialLibrary->Enabled = false;
        btnScan->Enabled = false;
        lbMaterials->BevelKind = bkFlat;
        lbMaterials->Enabled = false;
        DragAcceptFiles(pnlMaterials->Handle, false);
        }  // if (Status == statusDisabled)
      else {
        lblMaterialPanelCaptionShadow->Visible = false;
        lblMaterialPanelCaptionHighlight->Visible = false;
        edtMaterialLibrary->BevelKind = bkSoft;
        edtMaterialLibrary->Enabled = true;
        btnScan->Enabled = true;
        lbMaterials->BevelKind = bkSoft;
        lbMaterials->Enabled = true;
        DragAcceptFiles(pnlMaterials->Handle, true);
        if (Status == statusCurrent) {
          pnlMaterials->Font->Style = TFontStyles() << fsBold;
          // If this panel is current then the first two are enabled
          UpdatePanel(panelFile, statusEnabled);
          UpdatePanel(panelLOD, statusEnabled);
        }  // if (Status == statusCurrent)
        else {
          pnlMaterials->Font->Style = TFontStyles();
        }  // else (Status != statusCurrent)
      }  // else (Status != statusDisabled)
      break;
  }  // switch (TPanelIdent)
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::UpdatePanel(TPanelIdent Panel, TPanelStatus Status)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// checkScaleClick()
//
// checkScale OnClick event handler
// Enable/Disable the scale spin edit controls
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::checkScaleClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  spinX->Enabled = checkScale->Checked;
  spinY->Enabled = checkScale->Checked;
  spinZ->Enabled = checkScale->Checked;
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::checkScaleClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// checkExtractClick()
//
// checkExtract OnClick event handler
// If this checkbox is selected, then we need to export all the textures.
// We get a folder to export to, and then allow the user to select the
// export type of the texture.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::checkExtractClick(TObject *Sender)
{
  __ENTERFUNCTION__;
  if (checkExtract->Checked) {
    BROWSEINFO BrowseInfo;
    char Folder[MAX_PATH];
    LPITEMIDLIST IDList;

    BrowseInfo.hwndOwner      = Handle;
    BrowseInfo.pidlRoot       = NULL;
    BrowseInfo.pszDisplayName = Folder;
    BrowseInfo.lpszTitle      = "Select folder to save textures...";
    BrowseInfo.ulFlags        = BIF_RETURNONLYFSDIRS | BIF_USENEWUI;
    BrowseInfo.lpfn           = NULL;
    BrowseInfo.lParam         = 0;
    BrowseInfo.iImage         = 0;

    IDList = SHBrowseForFolder(&BrowseInfo);
    if (IDList) {
      SHGetPathFromIDList(IDList, Folder);
      TextureDir = Folder;
      radioTarga->Enabled = true;
      radioBMP->Enabled   = true;
      radioJPEG->Enabled  = true;
      checkLeaveDDS->Enabled = true;
    }  // if (IDList)
    else {
      checkExtract->Checked = false;
      checkExtractClick(Sender);
    }  // else (!IDList)
  }  // if (checkExtract->Checked)
  else {
    TextureDir = "";
    radioTarga->Enabled = false;
    radioBMP->Enabled   = false;
    radioJPEG->Enabled  = false;
    checkLeaveDDS->Enabled = false;
  }  // else (!checkExtract->Checked)
  __LEAVEFUNCTION__;
}  // void __fastcall TCMPImporter::checkExtractClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// btnOKClick()
//
// btnOK OnClick event handler
// This is what actually does the transfer of the imported model into
// MilkShape.
//---------------------------------------------------------------------------
void __fastcall TCMPImporter::btnOKClick(TObject *Sender)
{
  __ENTERFUNCTION__;

  int TriangleIndex = 0;
  int StartMaterial;

  msModel_SetTotalFrames(Model, 30);
  msModel_SetFrame(Model, 1);

  AnsiString  LoadedLibrary = "";
  TUTFFile   *MaterialLibrary;

  // Set up the materials.  Here we add the materials to
  // MilkShape, and we also extract the individual textures
  // from the Freelancer Material Library.
  MaterialLibrary = new TUTFFile;
  for (int n = 0; n < VMesh->Materials->Count; n++) {
    TMaterial  *Material = VMesh->Materials->Items[n];
    int         msMaterialID = msModel_AddMaterial(Model);
    msMaterial *msMaterial = msModel_GetMaterialAt(Model, msMaterialID);
    msVec4      Ambient  = { 0.2f, 0.2f, 0.2f, 1.0f };
    msVec4      Diffuse  = { 0.6f, 0.6f, 0.6f, 1.0f };
    msVec4      Specular = { 0.0f, 0.0f, 0.0f, 1.0f };
    msVec4      Emissive = { 0.0f, 0.0f, 0.0f, 0.0f };
    float       Shininess = 0.0f;
    float       Transparency = 1.0f;

    msMaterial_SetName(msMaterial, Material->Name.c_str());
    msMaterial_SetAmbient(msMaterial, Ambient);
    msMaterial_SetDiffuse(msMaterial, Diffuse);
    msMaterial_SetSpecular(msMaterial, Specular);
    msMaterial_SetEmissive(msMaterial, Emissive);
    msMaterial_SetShininess(msMaterial, Shininess);
    msMaterial_SetTransparency(msMaterial, Transparency);

    if (!Material->FileName.IsEmpty())
      if (checkExtract->Checked) {
        int TextureIndex;
        AnsiString SourceExt, DestExt;
        AnsiString SourceFileName, DestFileName;

        if (radioBMP->Checked)
          DestExt = ".BMP";
        else if (radioJPEG->Checked)
          DestExt = ".JPG";
        else
          DestExt = ".TGA";

        // See if we have this Material Library loaded already.  If so,
        // don't load it again.  Really primitive caching. :)
        if (Material->Library != LoadedLibrary) {
          MaterialLibrary->LoadFromFile(Material->Library);
          LoadedLibrary = Material->Library;
        }  // if (Material->Library != LoadedLibrary)
        TextureIndex = MaterialLibrary->IndexOf(String("\\texture library\\") + Material->FileName + String("\\MIPS"));
        if (TextureIndex == -1) {
          TextureIndex = MaterialLibrary->IndexOf(String("\\texture library\\") + Material->FileName + String("\\MIP0"));
          if (TextureIndex == -1) {
            msMaterial_SetDiffuseTexture(msMaterial, Material->FileName.c_str());
            continue;
          }  // if (TextureIndex == -1) inner
          else
            SourceExt = ".TGA";
        }  // if (TextureIndex != -1) outer
        else
          SourceExt = ".DDS";
        if (checkLeaveDDS->Checked) {
          SourceFileName = TextureDir + String("\\") + Material->FileName;
          SourceFileName = ChangeFileExt(Material->FileName, SourceExt);
        }  // if (checkLeaveDDS->Checked)
        else
          SourceFileName = TempFileName(SourceExt);
        MaterialLibrary->Items[TextureIndex]->SaveToFile(SourceFileName);
        DestFileName = ChangeFileExt(TextureDir + String("\\") + Material->FileName, DestExt);
        if (SourceExt != DestExt)
          ConvertImage(SourceFileName, DestFileName);
        else
          CopyFile(SourceFileName.c_str(), DestFileName.c_str(), false);
        msMaterial_SetDiffuseTexture(msMaterial, DestFileName.c_str());
        if (!checkLeaveDDS->Checked)
          DeleteFile(SourceFileName);
      }  // if (checkExtract->Checked)
      else {
        msMaterial_SetDiffuseTexture(msMaterial, Material->FileName.c_str());
      }  // else (!checkExtract->Checked)
    if (!n)
      StartMaterial = msMaterialID;
  }  // for (int n = 0; n < VMesh->Materials->Count; n++)
  delete MaterialLibrary;

  if (VMesh->Meshes->Count && !VMesh->MeshGroups->Count)
    VMesh->MeshGroups->Add("Root", 0, VMesh->Meshes->Count,0, VMesh->Vertices->Count - 1, 0, VMesh->Triangles->Count / 3 - 1);
  for (int MeshGroupLoop = 0; MeshGroupLoop < VMesh->MeshGroups->Count; MeshGroupLoop++) {
    TMeshGroup *MeshGroup = VMesh->MeshGroups->Items[MeshGroupLoop];

    for (int MeshLoop = MeshGroup->StartMesh; MeshLoop < MeshGroup->StartMesh + MeshGroup->MeshCount; MeshLoop++) {

      TMesh *Mesh = VMesh->Meshes->Items[MeshLoop];
      int msMeshID = msModel_AddMesh(Model);
      msMesh *msMesh = msModel_GetMeshAt(Model, msMeshID);

      msMesh_SetName(msMesh, (String("Group") + String(MeshLoop + 1)).c_str());
      msMesh_SetMaterialIndex(msMesh, VMesh->Materials->IndexOf(Mesh->MaterialID) + StartMaterial);

      for (int n = Mesh->StartVertex; n <= Mesh->EndVertex; n++) {
        int       VertexID = msMesh_AddVertex(msMesh);
        msVertex *Vertex   = msMesh_GetVertexAt(msMesh, VertexID);
        int       NormalID = msMesh_AddVertexNormal(msMesh);
        TVertex  *vmVertex = VMesh->Vertices->Items[n];
        msVec3    Coords;

        float XScale, YScale, ZScale;
        if (checkScale->Checked) {
          XScale = ((float)spinX->Value) / 100.0f;
          YScale = ((float)spinY->Value) / 100.0f;
          ZScale = ((float)spinZ->Value) / 100.0f;
        }  // if (checkScale->Checked)
        else {
          XScale = YScale = ZScale = 1.0f;
        }  // else (!checkScale->Checked)
        Coords[0] = (vmVertex->X + MeshGroup->OriginX) * XScale;
        Coords[1] = (vmVertex->Y + MeshGroup->OriginY) * YScale;
        Coords[2] = (vmVertex->Z + MeshGroup->OriginZ) * ZScale;

        msVertex_SetFlags(Vertex, 0);
        msVertex_SetVertex(Vertex, Coords);
        msVertex_SetTexCoords(Vertex, &vmVertex->U);
        msVertex_SetBoneIndex(Vertex, -1);

        msMesh_SetVertexNormalAt(msMesh, NormalID, &vmVertex->NormalX);
      }  // for (int n = 0; n < VMesh->Vertices->Count; n++)

      for (int n = Mesh->StartTriangle; n <= Mesh->EndTriangle; n++) {
        int         TriangleID = msMesh_AddTriangle(msMesh);
        msTriangle *Triangle   = msMesh_GetTriangleAt(msMesh, TriangleID);
        TTriangle  *vmTriangle = VMesh->Triangles->Items[n + TriangleIndex];

        WORD Indices[3];

        Indices[0] = vmTriangle->Vertex1;
        Indices[1] = vmTriangle->Vertex2;
        Indices[2] = vmTriangle->Vertex3;

        msTriangle_SetFlags(Triangle, 0);
        msTriangle_SetVertexIndices(Triangle, Indices);
        msTriangle_SetNormalIndices(Triangle, Indices);
        msTriangle_SetSmoothingGroup(Triangle, 1);
      }  // for (int n = 0; n < VMesh->Triangles->Count; n++)
    }  // for (int MeshLoop = 0; MeshLoop < VMesh->Meshes->Count; MeshLoop++)
  }  // for (int MeshGroupLoop = 0; MeshGroupLoop < VMesh->MeshGroups->Count; MeshGroupLoop++)
  __LEAVEFUNCTION__;
  ModalResult = mrOk;
}  // void __fastcall TCMPImporter::btnOKClick(TObject *Sender)
//---------------------------------------------------------------------------




