#include "ms3d.h"
#include "utils.h"
#include "ms3dspec.h"
#include "scenenum.h"
#include "decomp.h"
#include "joint.h"


//--------------------------
// CScene class code.
//--------------------------

using namespace std;

CScene::CScene(Interface *_iface)
{
	// Default stuff.
	animationFPS = 24;
	currentTime = 0;
	totalFrames = 30;

	iface = _iface;


	ticksPerFrame = GetTicksPerFrame();
	interval = iface->GetAnimRange();
	float totalTicks = (float)(interval.End() - interval.Start());
	animationFPS = (float)GetFrameRate();
	totalFrames = (int)(totalTicks / (float)ticksPerFrame);
	totalTime = totalFrames / animationFPS ; // animation length (in seconds).
	currentTime = 0; 

}

CScene::~CScene()
{
	for(vector<ms3d_triangle_t *>::iterator i = face.begin(); i != face.end(); i++)
		delete *i;
	for(vector<ms3d_Vector3_t *>::iterator j = vertex.begin(); j != vertex.end(); j++)
		delete *j;
	for(vector<CJoint *>::iterator k = joint.begin(); k != joint.end(); k++)
		delete *k;
	
	for(vector<ms3d_group_t *>::iterator g = group.begin(); g != group.end(); g++) {
		delete[] (*g)->triangleIndex;
		delete *g;
	}
}


ms3d_material_t GetDefaultMaterial() 
{
	ms3d_material_t mat;
	strcpy(mat.name, "Material01");
	SetVec4(mat.diffuse, 0.16f, 0.43f, 0.52f, 1); 
	SetVec4(mat.ambient, 0.8f, 0.8f, 0.8f, 1);
	SetVec4(mat.specular, 0, 0, 0, 1);
	SetVec4(mat.emissive, 0, 0, 0, 1);
	mat.shininess = 0;
	mat.transparency = 1;
	mat.mode = 2;
	strcpy(mat.texture, "");
	strcpy(mat.alphamap, "");
	return mat;
}


ms3d_material_t CScene::GetMaterial(char index)
{
	ms3d_material_t out;
	memset(&out, 0, sizeof(ms3d_material_t));
	TimeValue time = interval.Start();
	
	if(index < 0 || index > (int)material.size()) {
		MessageBox(NULL, "index out of bounds!", "GetMaterial", MB_OK|MB_ICONERROR);
		return out;
	}

	Mtl *mat = material[index];


	// See if it's a Standard material
	if(mat->ClassID() == Class_ID(DMTL_CLASS_ID, 0)) { 
		StdMat* std = (StdMat *)mat;

		strcpy(out.name, mat->GetName());

		Color d = std->GetDiffuse(time);
		Color a = std->GetAmbient(time);
		Color s = std->GetSpecular(time);
			
		SetVec4(out.diffuse, d.r, d.g, d.b, 1); 
		SetVec4(out.ambient, a.r, a.g, a.b, 1); 
		SetVec4(out.specular, s.r, s.g, s.b, 1); 
		SetVec4(out.emissive, 0, 0, 0, 1); // MAKEME: do this.. self illumination functions...
		out.transparency = std->GetOpacity(time);
		out.mode = 2;


		// This is wierd.. milkshape seems to be inverse (ie the higher shinyness the sharper 'smalldot' effect on the specular). 
		// MAKEME: gloss and stuff? , figure out why shinestr can go up to 10!
		float str = 1.0f - std->GetShinStr(time);
		if(str < 0)
			str = 0;
		out.shininess = 128 * str; 


		// Defaults... 
		strcpy(out.texture, "");
		strcpy(out.alphamap, "");


		Texmap *tmap;
		TCHAR *file;			

		tmap = mat->GetSubTexmap(ID_DI); // // Use diffuse as texture.
		if (tmap && tmap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0)) {
			BitmapTex *bmt = (BitmapTex*) tmap;
			file = bmt->GetMapName();
			if(file) {
				strcpy(out.texture, file);
				SetVec4(out.diffuse, 1, 1, 1, 1);  // Clear diffuse color (the texture does the job from now on)...
			}
		}

		tmap = mat->GetSubTexmap(ID_OP); // // Use opacity map as alhpa.
		if (tmap && tmap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0)) {
			BitmapTex *bmt = (BitmapTex*) tmap;
			file = bmt->GetMapName();
			if(file)
				strcpy(out.alphamap, file);
		}


		stripPath(out.alphamap);
		stripPath(out.texture);

	}
	else
		return GetDefaultMaterial();

	return out;
}



void CScene::Write(const char *filename) 
{
	int i;
	FILE *fp;


	fp = fopen(filename, "wb");
	if(!fp) {
		return;
	}


	// Write the header.
	ms3d_header_t header;
	strcpy(header.id, "MS3D000000");
	header.version = 4;
	fwrite(&header, sizeof(ms3d_header_t), 1, fp);


	// Write vertices
	word numverts = vertex.size();
	fwrite(&numverts, 2, 1, fp); 
	for(i = 0; i < numverts; i++) 
		fwrite(vertex[i], sizeof(ms3d_Vector3_t), 1, fp); 

	// Write faces
	word numfaces = face.size();
	fwrite(&numfaces, 2, 1, fp); 
	for(i = 0; i < numfaces; i++) 
		fwrite(face[i], sizeof(ms3d_triangle_t), 1, fp); 
	

	word numGroups = group.size();
	fwrite(&numGroups, 2, 1, fp); 

	for(i = 0; i < numGroups; i++) {
		ms3d_group_t g = *group[i];

		fwrite(&g, 1 + 32 + 2, 1, fp); 
		for(int j = 0; j < g.numtriangles; j++)
			fwrite(&g.triangleIndex[j], 2, 1, fp); 
		fwrite(&g.materialIndex, 1, 1, fp); 
	}

	word numMaterials = material.size();

	// If there's no materials we have to make a default...
	if(!numMaterials) {
		numMaterials = 1;
		fwrite(&numMaterials, 2, 1, fp); 
		ms3d_material_t defmat = GetDefaultMaterial();
		fwrite(&defmat, sizeof(ms3d_material_t), 1, fp); 
	}
	else {
		fwrite(&numMaterials, 2, 1, fp); 
		for(i = 0; i < numMaterials; i++) { 
			ms3d_material_t mat = GetMaterial(i);
			fwrite(&mat, sizeof(ms3d_material_t), 1, fp); 
		}
	}


	// save some keyframer data
	fwrite(&animationFPS, sizeof(float), 1, fp); 
	fwrite(&currentTime, sizeof(float), 1, fp); 
	fwrite(&totalFrames, sizeof(int), 1, fp); 

	// Write joint data.
	word numJoints = joint.size();
	fwrite(&numJoints, 2, 1, fp);
	
	for(i = 0; i < numJoints; i++) {
		long size = joint[i]->Size();
		char *temp = new char[size];

		joint[i]->FillByteBuffer(temp);

		fwrite(temp, size, 1, fp); 
		// MAKEME: write the joint keyframer data.			
		delete[] temp;
	}

	fclose(fp);
}


// Returns index of parent for joint[id]
// or -1 if no parent.
int CScene::GetParentIndex(int id)
{
	int i;
	for(i = 0; i < (int)joint.size(); i++) {
		if(!strcmp(joint[id]->parentName, joint[i]->name)) {
			return i;
		}
	}
	return -1;
}


int CScene::GetNumChildren(INode *j) 
{
	int count = 0;
	for(int i = 0; i < (int)joint.size(); i++) {
	
		if(joint[i]->parent == j)
			count ++;			
	}
	return count;
}

int CScene::GetIndex(INode *j)
{
	for(int i = 0; i < (int)joint.size(); i++) {
	
		if(joint[i]->node == j)
			return i;
	}
	return -1;
}

CJoint* CScene::GetJoint(INode *node)
{
	for(int i = 0; i < (int)joint.size(); i++) {
	
		if(joint[i]->node == node)
			return joint[i];
	}
	return NULL;
}

// Recursive stuff.
INode *CScene::GetRedir(INode *from)
{
	for(int r = 0; r < (int)jointRedirect.size(); r++) {
		if(from  == jointRedirect[r].from) {
			return GetRedir(jointRedirect[r].to);
		}
	}
	return from; // No more redirections found.
}


void CScene::SetupJoints()
{
	int i, j;
	bool done = false;
	CJoint *tmp;
	int parentIndex;



	const int numJoints = joint.size();

	// Find parents for each joint.
	for(i = 0; i < numJoints; i++) {
		joint[i]->parent = GetRedir(joint[i]->parent);
		if(joint[i]->parent)
			strncpy(joint[i]->parentName, joint[i]->parent->GetName(), 32); 
		else
			strncpy(joint[i]->parentName, "", 32); 
	}

	// sort the joints in the milkshape fashion.
	// the parent of any joint must be earlier in the vector than that joint.
	for(i = 1; i < numJoints; i++) {
		done = true;
		for(j = 0; j < numJoints; j++) {
			if(strcmp(joint[i]->parentName, joint[i - 1]->name) ) { // if not the joint[i - 1] is parent to joint[i]. swap em.
				tmp = joint[i];
				joint[i] = joint[i - 1];
				joint[i - 1] = tmp;
				done = false;
			}
		}
		if(done)
			break;
	}

	// Calculate length's between each joint and it's parent.
	// And set the number of childrens..
	for(i = 0; i < numJoints; i++) {
		parentIndex = GetParentIndex(i);
		if(parentIndex >= 0) {
			Point3 p1(joint[i]->position), p2(joint[parentIndex]->position);
			joint[i]->lengthToParent = (p1 - p2).Length(); 
		}
		else
			joint[i]->lengthToParent = -1;

		joint[i]->numChildren = GetNumChildren(joint[i]->node);
	}

	// Find jointId for each vertex.
	 // valid physicque modifier exist.
	for(i = 0; i < (int)vertex.size(); i++) {
		char *assign;
		vertex[i]->jointId = -1; // assume no joint.

		assign = (char *)jointName[i];

		INode *redir = GetRedir(iface->GetINodeByName(assign));
		if(redir)
			assign = redir->GetName();
		
		// Loop through all joints to find matching joint.
		//for(vector<ms3d_joint_t*>::iterator ji = joint.begin(); ji != joint.end(); ji++) {
		for(j = 0; j < numJoints; j++) {
			if(!strncmp(joint[j]->name, assign, 32)) {
				vertex[i]->jointId = j;
				break;
			}
		}
	}

	for(i = 0; i < numJoints; i++) {
		joint[i]->GetTransformation(interval.Start(), joint[i]->rotation, joint[i]->position);
	
		if(expOpt.anim) {
			joint[i]->GetKeyframes();
			joint[i]->FinalizeKeys();
			joint[i]->numKeyFramesRot = joint[i]->keyFramesRot.size();
			joint[i]->numKeyFramesTrans = joint[i]->keyFramesTrans.size();
		}
	}
}


void CScene::ProcessJoint(INode *node)
{
	CJoint *j = new CJoint;
	int i;

	j->node = node;
	j->pos = node->GetNodeTM(0).GetTrans(); 
	strncpy(j->name, node->GetName(), 32);
	j->flags = 0;
	j->scene = this;

	
	j->parent = node->GetParentNode();
	if(j->parent && !IsBone(j->parent)) // have to check that it's really a bone (could be scene root).
		j->parent = NULL;

	if (expOpt.selectedOnly && !node->Selected()) { // Don't include if not selected.
		joint_rd_t redir; // Redirect any children to this joint to it's parent
		redir.from = j->node;
		redir.to = j->parent;
		jointRedirect.push_back(redir);
//		note("excluded %s\n", j->name);
		delete j;
		return;
	}

	// MAX is keff! if two bones are attached to the same parent. MAX chooses to create two joints at the intersection!!.
	// Below code checks for dublicates.

	const float EPSILON = 0.0005f; // distance limit to another joint before joint is considered dublicate.
	for(i = 0; i < (int)joint.size(); i++) {

		if((joint[i]->pos - j->pos).Length() < EPSILON) {
			//note("%s == %s (%s removed)", joint[i]->name, j->name, j->name);

			joint_rd_t redir;
			redir.from = j->node;
			redir.to = joint[i]->node;
			jointRedirect.push_back(redir);

			delete j;
			return;
		}
	}
	joint.push_back(j);
}


void CScene::ProcessGeometricObject(Object *obj, INode *node)
{
	TimeValue time = interval.Start();
	int i;
	char matIndex;
	TriObject *tri = (TriObject *)obj->ConvertToType(time, triObjectClassID);

	if(!tri) {
		MessageBox(NULL, "Could not get triangle mesh!, skipping object", node->GetName(), MB_OK|MB_ICONWARNING);
		return;
	}

	Mesh &mesh = tri->GetMesh();


	// Indexes must be adjusted with respect to geometric objects already added to this scene.
	word vertexOffset = vertex.size();
	word faceOffset = face.size();
	int jointNameOffset = jointName.size();

	int numVerts = mesh.getNumVerts();
	int numFaces = mesh.getNumFaces();

	// Check vertex & face limits.
	if(vertexOffset + numVerts >= 65535) {
		MessageBox(NULL, "Vertexcount limit reached!, skipping object", node->GetName(), MB_OK|MB_ICONWARNING);
		return;
	}
	if(faceOffset + numFaces >= 65535) {
		MessageBox(NULL, "Facecount limit reached!, skipping object", node->GetName(), MB_OK|MB_ICONWARNING);
		return;
	}

	ms3d_group_t *g = new ms3d_group_t;



	// Get material for this node.
	Mtl *mat = node->GetMtl();
	if(mat && expOpt.materials) {
		matIndex = -1;
		// Search if we already has added this material...
		for(i = 0; i < (int)material.size(); i++) {
			if(material[i] == mat) {
				matIndex = i;
				break;
			}
		}
		if(matIndex < 0) { // Haven't added before so do it now.
			matIndex = material.size(); // this group will have the index for the newly added material.
			material.push_back(mat);
		}
	}
	else
		matIndex = 0;

	// Do the vertices.
	// process each vertex in the mesh.
	for(i = 0; i < numVerts; i++) {
		ms3d_Vector3_t *vert = new ms3d_Vector3_t;
		Point3 p = mesh.getVert(i);		

		//Matrix3 m = node->GetNodeTM(0); // get mesh at start time... (animation is only done with bones)
		Matrix3 m = node->GetObjectTM(interval.Start()); // FIX!, we need the WORLD space coords, not object space or what is was !)

		p = p * m * ms3d;

		vert->flags = 0;
		vert->jointId = -1; // no joint MAKEME: solve...
		vert->referenceCount = 0; // (dont know what this is)

		vert->Vector3[0] = p.x;
		vert->Vector3[1] = p.y;
		vert->Vector3[2] = p.z;
		vertex.push_back(vert);

		// Always keep jointName up to date with vertex count.
		Str32 tmp;
		tmp = ""; // no joint...
		jointName.push_back(tmp);
	}


//	mesh.RemoveDegenerateFaces(); //Removes faces that have two or more equal indices.
//	mesh.RemoveIllegalFaces(); //Removes faces that have indices that are out of range

	for(i = 0; i < numFaces; i++) {
		ms3d_triangle_t *t = new ms3d_triangle_t;
		Face f = mesh.faces[i];

		if(mesh.getNumTVerts() > 0)
		{
			TVFace tvFace = mesh.tvFace[i];
			for(int j = 0; j < 3; j++) {
				UVVert tCoord;
				int index = tvFace.getTVert(j);
				if(index > mesh.getNumTVerts()) {
					MessageBox(NULL, "Found invalid tCoord index!", node->GetName(), MB_OK|MB_ICONERROR);
					break;
				}
				tCoord = mesh.getTVert(index);
				t->s[j] = tCoord.x;
				t->t[j] = 1 - tCoord.y;
			}
		}
		else {
//			if(!i) // only show this one time (not one for every face in object).
//				MessageBox(NULL, "Note: Object has no texture coordinates.", node->GetName(), MB_OK);
			memset(t->s, 0, sizeof(float) * 3);
			memset(t->t, 0, sizeof(float) * 3);
		}
		

		t->smoothingGroup = (byte)f.getSmGroup(); // we'll loose half.. =/
		
		for(int j = 0; j < 3; j++) {
			word index = (word)(f.v[j] + vertexOffset);
			t->Vector3Indices[j] = index;
		}

		t->flags = 0;
		t->groupIndex = group.size();
		face.push_back(t);
	}

	// Setup this object as a group.
	strncpy(g->name, node->GetName(), 32); 
	g->triangleIndex = new word[numFaces];
	for(i = 0; i < numFaces; i++)
		g->triangleIndex[i] = faceOffset + i;	
	g->numtriangles = numFaces;
	g->flags = 0;
	g->materialIndex = matIndex;
	group.push_back(g); // add to vector.

	
	if(expOpt.bones) {
		Modifier *phyMod = GetPhysiqueModifier(node); // Check if node has a physique modifer somewhere in the pipeline of it's derived objects.
		Modifier *skinMod = GetSkinModifier(node); // Check if node has a physique modifer somewhere in the pipeline of it's derived objects.

		if(skinMod)
			ProcessSkinData(node, skinMod, jointNameOffset);		
		else if(phyMod)
			ProcessPhysiqueData(node, phyMod, jointNameOffset);		
	}

//	MessageBox(NULL, "Object finished", node->GetName(), MB_OK);
}


// Returns true if the vertex is 
inline bool CScene::VertexInFace(int vert, int f)
{
	int i;

	for(i = 0; i < 3; i++) {
		if(face[f]->Vector3Indices[i] == vert)
			return true;
	}
	return false;
}

// Calculates the normals for each face in the mesh.
// MAKEME: Optimize!!  this fucker is really slow (probably stupid ?=).

void CScene::CalculateNormals()
{
	int i, j, k;
	const int numTriangles = face.size();
	Point3 a, b, c, n;

	if(!numTriangles) { // model has no faces (could be only joints)
		return;		
	}

	Point3 *normal = new Point3[numTriangles];

	// go through each face and calulate flatshaded normal..
	for(i = 0; i < numTriangles; i++) {
		a = Point3(vertex[face[i]->Vector3Indices[0]]->Vector3);
		b = Point3(vertex[face[i]->Vector3Indices[1]]->Vector3);
		c = Point3(vertex[face[i]->Vector3Indices[2]]->Vector3);
		normal[i] = (a - c) ^ (b - c); // calulate normal with crossproduct.
		normal[i] = Normalize(normal[i]);
	}

	if(expOpt.noSmooth) {
		for(i = 0; i < numTriangles; i++) {
			for(j = 0; j < 3; j++) { // for each vertex in face.
				face[i]->Vector3Normals[j][0] = normal[i].x;
				face[i]->Vector3Normals[j][1] = normal[i].y;
				face[i]->Vector3Normals[j][2] = normal[i].z;
			}
		}
		delete[] normal;
		return;
	}
	// else continue with smoth shading....

	// Smooth shading.
	float len;
	for(i = 0; i < numTriangles; i++) {

		for(j = 0; j < 3; j++) { // for each vertex in face.

			n.x = 0;
			n.y = 0;
			n.z = 0;

			// Find each face that this vertex is part of. Add their normals.
			for(k = 0; k < numTriangles; k++) {
				if(VertexInFace(face[i]->Vector3Indices[j], k) && face[k]->smoothingGroup == face[i]->smoothingGroup)
					n += normal[k];
			}

			//n = Normalize(n);
			len = sqrt((n.x * n.x) + (n.y * n.y) + (n.z * n.z));
			n.x /= len;
			n.y /= len;
			n.z /= len;

			face[i]->Vector3Normals[j][0] = n.x;
			face[i]->Vector3Normals[j][1] = n.y;
			face[i]->Vector3Normals[j][2] = n.z;
		}
	}

	// Cleanup.
	delete[] normal;		
}


bool CScene::ProcessPhysiqueData(INode* node, Modifier* mod, int offset)
{
	INode *bone;
	TSTR name;

	//check to make sure it is physique
	if (!mod || mod->ClassID() != Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
		return false;

	//get a pointer to the export interface
	IPhysiqueExport *phyExport = (IPhysiqueExport *)mod->GetInterface(I_PHYEXPORT);

	//get the physique version number.  
	//If the version number is > 30 you may have floating bones
	int ver = phyExport->Version();

	//get the node's initial transformation matrix and store it in a matrix3
	Matrix3 initTM;
	int msg = phyExport->GetInitNodeTM(node, initTM);

	//get a pointer to the export context interface
	IPhyContextExport *mcExport = (IPhyContextExport *)phyExport->GetContextInterface(node);

	//convert to rigid for time independent vertex assignment
	mcExport->ConvertToRigid(true);

	mcExport->AllowBlending(true); // (export them as rigid)
	
	//get the vertex count from the export interface
	int numverts = mcExport->GetNumberVertices();

	// Check for inconsistency in the number of vertices of the mesh that got this modifier applied (node), and it's current mesh.
	if(numverts != (int)vertex.size() - offset) {

		// Can't have more vertices attached to bones than exists in the mesh object.
		phyExport->ReleaseContextInterface(mcExport);
		mod->ReleaseInterface(I_PHYINTERFACE, phyExport);
		MessageBox(NULL, "Vertex number mismatch! Ignoring physique modifier.\nResult may be undefined.", "Warning", MB_OK|MB_ICONERROR);

		return false;
	}
//	if(numverts < (int)vertex.size()) 
//		MessageBox(NULL, "There's unassigned vertices in the scene.", "Warning", MB_OK|MB_ICONWARNING);

	bool supress = false;

	//gather the vertex-link assignment data
	for (int i = 0; i < numverts; i++)
	{

		//Get the hierarchial vertex interface for this vertex
		IPhyVertexExport* vi = mcExport->GetVertexInterface(i);
		if (vi){

			//check the vertex type and process accordingly	
			IPhyRigidVertex *vrt;
			int type = vi->GetVertexType();
			switch (type) {
				//we have a vertex assigned to more than one link
				case RIGID_BLENDED_TYPE: 
					if(!supress) // only warn one time.
						supress = (MessageBox(NULL, "Found blended vertices!, Not allowed in the ms3d format-> These vertices will be ignored in the physique modifier. Result may be undefined.\nSuppress further error messages?", "Warning", MB_YESNO|MB_ICONWARNING) == IDYES);
					break;					
				case RIGID_TYPE: // we have a vertex assigned to a single link
					//type-cast the node to the proper class
					vrt = (IPhyRigidVertex*)vi;
					bone = vrt->GetNode();
					name =  bone->GetName();
					jointName[i + offset] = name;

					//jointName[i + offset] = vrt->GetNode()->GetName();
					break;
				//Should not make it here since assignments were converted to Rigid.  
				//Should be one of the above two types
				default:
					MessageBox(NULL, "Found unknown vertex type! Result may be undefined.", "Warning", MB_OK|MB_ICONWARNING);
					break;
			}
			//release the vertex interface
			mcExport->ReleaseVertexInterface(vi);
		}
	}
	//release the context interface
	phyExport->ReleaseContextInterface(mcExport);
	//Release the physique interface
	mod->ReleaseInterface(I_PHYINTERFACE, phyExport);

	return true;
}


// DEBUG: make this stable, not throughtly tested...
bool CScene::ProcessSkinData(INode* node, Modifier* mod, int offset)
{

	int vertexId, i;

	//check to make sure modifier is a valid skin modifer.
	if (!mod || mod->ClassID() != SKIN_CLASSID)
		return false;

	ISkin *iskin = (ISkin *)mod->GetInterface(I_SKIN);

	if(!iskin)
		return false;

	ISkinContextData *cskin = iskin->GetContextInterface(node);
	if(!cskin)
		return false;

	int numverts = cskin->GetNumPoints();
	int numbones = iskin->GetNumBones(); // Number of bones that the modifier works with.
	
	if(numverts != (int)vertex.size() - offset) {
		// Can't have more  vertices attached to bones than exists in the mesh object.

		mod->ReleaseInterface(I_SKIN, iskin);
		note("Vertex number mismatch! Ignoring skin modifier.\nResult may be undefined.\nMesh has %d vertices\nModifier %d\n", (int)vertex.size() - offset, numverts);
		return false;
	}

	//gather the vertex-link assignment data
	for (vertexId = 0; vertexId < numverts; vertexId++)
	{
		int numAssignedBones = 	cskin->GetNumAssignedBones(vertexId);
		int boneIndex = -1;
		float maxWeight;
		
		for(i = 0; i < numAssignedBones; i++) {
			float weight = cskin->GetBoneWeight(vertexId, i);

			int index = cskin->GetAssignedBone(vertexId, i);

			
			//if(!iskin->GetBoneName(index)) // If there are any unnamed bones out there!! (can be).
			//	continue;
			
			// Get the bone which has most influence on this vertex.
			if(boneIndex < 0 || weight > maxWeight) {
				boneIndex = index;
				maxWeight = weight;
			}
		}

		if(boneIndex < 0 || boneIndex > numbones - 1) {
			note("%s Skinmodifier:\nVertex %d has invalid bone index (%d / %d)\nNot assigning vertex to any bone. Results may be undefined!", node->GetName(), vertexId, boneIndex, numbones);
			continue;			
		}


		char *bone = iskin->GetBoneName(boneIndex);

		if(!bone)
			note("<%s>\nSkinmodifier:\nVertex %d has invalid bone name!\nboneIndex=%d\n(should never happen)", node->GetName(), vertexId, boneIndex);
		else
			jointName[vertexId + offset] = bone;
	}

	mod->ReleaseInterface(I_SKIN, iskin);
	return true;
}
