#include "ms3d.h"
#include "utils.h"
#include "ms3dspec.h"
#include "decomp.h"
#include "joint.h"

using namespace std;

//------------------------------
// Joint methods (class CJoint
//------------------------------

CJoint::CJoint()
{
	numKeyFramesRot = 0;
	numKeyFramesTrans = 0;
}

CJoint::~CJoint()
{
	vector<ms3d_keyframe_t *>::iterator i;
	for(i = keyFramesRot.begin(); i != keyFramesRot.end(); i++)
		delete *i;
	for(i = keyFramesTrans.begin(); i != keyFramesTrans.end(); i++)
		delete *i;
}


// As INode::GetNodeTM() but the matrix is given in ms3d coordinates.
Matrix3 CJoint::GetMs3dTM(TimeValue time)
{
	Matrix3 m = node->GetNodeTM(time);
	m.NoScale();
	return m * ms3d;
}

Matrix3 CJoint::GetParentMs3dTM(TimeValue time)
{
	Matrix3 m;
	if(parent) 
		m = parent->GetNodeTM(time);
	else
		return IdentityTM();
	m.NoScale();
	return m * ms3d;
}


// Used to get start (reference) translation & rotation of joint.
void CJoint::GetTransformation(const TimeValue& time, float *rot, float *trans)
{
	// Note: timevalue must be in ticks.
	Matrix3 relative;
	Point3 pos;

	Matrix3 tm = GetMs3dTM(time);
	Matrix3 ptm	= GetParentMs3dTM(time);
	relative = tm  * Inverse(ptm);
	pos = relative.GetTrans();

	if(rot) {
		// Get the rotation (via decomposition into "affine parts", then quaternion-to-Euler)
		// Apparently the order of rotations returned by QuatToEuler() is X, then Y, then Z.
		AffineParts affparts;
		decomp_affine(relative, &affparts);
		QuatToEuler(affparts.q, rot);
		
		// Get rotations in the -2pi...2pi range
		rot[0] = ReduceRotation(rot[0]);
		rot[1] = ReduceRotation(rot[1]);
		rot[2] = ReduceRotation(rot[2]);
	}
	if(trans) {
		trans[0] = pos.x;
		trans[1] = pos.y;
		trans[2] = pos.z;
	}
}


// get the difference between a & b (used for keyframing).
void CJoint::GetTransformation(const TimeValue& a, const TimeValue& b, float *rot, float *trans)
{
	// Note: timevalue must be in ticks.
	Matrix3 relative;
	Point3 pos;

	Matrix3 rel_a = GetMs3dTM(a) * Inverse( GetParentMs3dTM(a) ); // relative matrix at time a.
	Matrix3 rel_b = GetMs3dTM(b) * Inverse( GetParentMs3dTM(b) ); // relative matrix at time a.
	relative = rel_b * Inverse(rel_a);
	pos = relative.GetTrans();

	if(rot) {
		// Get the rotation (via decomposition into "affine parts", then quaternion-to-Euler)
		// Apparently the order of rotations returned by QuatToEuler() is X, then Y, then Z.
		AffineParts affparts;
		decomp_affine(relative, &affparts);
		QuatToEuler(affparts.q, rot);
		// Get rotations in the -2pi...2pi range
		rot[0] = ReduceRotation(rot[0]);
		rot[1] = ReduceRotation(rot[1]);
		rot[2] = ReduceRotation(rot[2]);
	}
	if(trans) {
		trans[0] = pos.x;
		trans[1] = pos.y;
		trans[2] = pos.z;
	}
}


/* This is the NON strange version but its little fucked up. (check the foot node for exported bipeds).
// As INode::GetNodeTM() but the matrix is given in ms3d coordinates.
Matrix3 CJoint::GetMs3dTM(TimeValue time)
{
	Matrix3 m = node->GetNodeTM(time);
	m.NoScale();
	if(numChildren > 1)
		clearRotation(m);
	return m * ms3d;
}

Matrix3 CJoint::GetParentMs3dTM(TimeValue time)
{
	Matrix3 m;
	CJoint *p = scene->GetJoint(parent);
	
	if(p) {
		m = p->GetMs3dTM(time);
	}
	else {
		//MessageBox(NULL, "Joint got no parent!", "GetParentMs3dTM", MB_OK|MB_ICONERROR);
		return IdentityTM();
	}
	return m;
}


// Used to get start (reference) translation & rotation of joint.
void CJoint::GetTransformation(const TimeValue& time, float *rot, float *trans)
{
	// Note: timevalue must be in ticks.
	Matrix3 relative;
	Point3 pos;
	CJoint *pj;

	Matrix3 ptm; // parent translation matrix.
	Matrix3 tm = GetMs3dTM(time);

	ptm	= GetParentMs3dTM(time);
	relative = tm  * Inverse(ptm);
	
	pj = scene->GetJoint(parent); // Get CJoint from parent.
	if(pj && pj->numChildren == 1) {
		float len = (ptm.GetTrans() - tm.GetTrans()).Length();
		pos.Set(len, 0, 0);
	}
	else
		pos = relative.GetTrans();

	if(rot) {
		// Get the rotation (via decomposition into "affine parts", then quaternion-to-Euler)
		// Apparently the order of rotations returned by QuatToEuler() is X, then Y, then Z.
		AffineParts affparts;
		decomp_affine(relative, &affparts);
		QuatToEuler(affparts.q, rot);
		
		// Get rotations in the -2pi...2pi range
		rot[0] = ReduceRotation(rot[0]);
		rot[1] = ReduceRotation(rot[1]);
		rot[2] = ReduceRotation(rot[2]);
	}
	if(trans) {
		trans[0] = pos.x;
		trans[1] = pos.y;
		trans[2] = pos.z;
	}
}


// get the difference between a & b (used for keyframing).
void CJoint::GetTransformation(const TimeValue& a, const TimeValue& b, float *rot, float *trans)
{
	// Note: timevalue must be in ticks.
	Matrix3 relative;
	Point3 pos;

	Matrix3 rel_a = GetMs3dTM(a) * Inverse( GetParentMs3dTM(a) ); // relative matrix at time a.
	Matrix3 rel_b = GetMs3dTM(b) * Inverse( GetParentMs3dTM(b) ); // relative matrix at time a.
	relative = rel_b * Inverse(rel_a);
	pos = relative.GetTrans();

	if(rot) {
		// Get the rotation (via decomposition into "affine parts", then quaternion-to-Euler)
		// Apparently the order of rotations returned by QuatToEuler() is X, then Y, then Z.
		AffineParts affparts;
		decomp_affine(relative, &affparts);
		QuatToEuler(affparts.q, rot);
		// Get rotations in the -2pi...2pi range
		rot[0] = ReduceRotation(rot[0]);
		rot[1] = ReduceRotation(rot[1]);
		rot[2] = ReduceRotation(rot[2]);
	}
	if(trans) {
		trans[0] = pos.x;
		trans[1] = pos.y;
		trans[2] = pos.z;
	}
}
*/



void CJoint::FinalizeKeys()
{
	int i;
	ms3d_keyframe_t *key;
	TimeValue start = scene->interval.Start();

/* NOTES:
	the max keys seems to be absolute (ie not dependent on bones original position).
	and they are local for each bone... = the rotation for a child doesnt change if the parent change.
	what need's to be done to convert to ms3d format is.

	1. find out the bones position & rotation relative to its starting position.
	2. swap coordinate system.
	
	the key->mod entries are actually unused. But are still needed when figuring out the times
	that the keyframes are set to.
*/

	for(i = 0; i < (int)keyFramesRot.size(); i++) {
		key = keyFramesRot[i];

		if(start != (TimeValue)key->time)
			GetTransformation(start, (TimeValue)key->time, key->mod, NULL);
		else
			memset(key->mod, 0, sizeof(float) * 3);

		key->time = scene->GetTime(key->time); // Convert ticks to secs.
	}

	for(i = 0; i < (int)keyFramesTrans.size(); i++) {
		key = keyFramesTrans[i];

		if(start != (TimeValue)key->time)
			GetTransformation(start, (TimeValue)key->time, NULL, key->mod);
		else
			memset(key->mod, 0, sizeof(float) * 3);

		key->time = scene->GetTime(key->time); // Convert ticks to secs.
	}

/*
	// DEBUG (output key info).
	char buf[1024];
	vector<ms3d_keyframe_t *> *vec;
	vec = &keyFramesRot; 
	strcpy(buf, "rotation keyframes:\n");

	for(i = 0; i < (int)vec->size(); i++) {
		char line[128];
		key = (*vec)[i];

		sprintf(line, "(%.2f): %.2f, %.2f, %.2f\n", key->time, key->mod[0], key->mod[1], key->mod[2]); 
		strcat(buf, line);
	}
	MessageBox(NULL, buf, name, MB_OK); 
*/
}

// return's the current size of this joint structure.
long CJoint::Size() 
{
	long size = sizeof(ms3d_joint_t);
	size += sizeof(ms3d_keyframe_t) * keyFramesRot.size();
	size += sizeof(ms3d_keyframe_t) * keyFramesTrans.size();
	return size;
}

// fills an array of bytes with the data for this joint (useful when writing).
void CJoint::FillByteBuffer(char *buf)
{
	int i;
	const int keysize = sizeof(ms3d_keyframe_t);
	long offset = sizeof(ms3d_joint_t);
	memcpy(buf, this, offset);

	for(i = 0; i < (int)keyFramesRot.size(); i++) {
		memcpy(buf + offset, keyFramesRot[i], keysize); 
		offset += keysize;
	}
	for(i = 0; i < (int)keyFramesTrans.size(); i++) {
		memcpy(buf + offset, keyFramesTrans[i], keysize); 
		offset += keysize;
	}
}


void CJoint::GetKeyframes()
{
	if(IsBipedBone(node)) {
		GetPRSKeys();
	}
	else {
		if(!GetPRSKeys())
			GetKeyframes(node);
	}
}


bool CJoint::GetPRSKeys()
{
	TimeValue time, next;
	ms3d_keyframe_t *key;
	DWORD flags;
	// Get the node's transform control
	Control *c = node->GetTMController();

	keyFramesRot.clear();
	keyFramesTrans.clear();

	flags = NEXTKEY_RIGHT | NEXTKEY_ROT;
	time = scene->interval.Start() - 1;
	while(c->GetNextKeyTime(time, flags, next)) {

		// looped?
		if(keyFramesRot.size() && next == keyFramesRot[0]->time)
			break;
		
		key = new ms3d_keyframe_t;
		key->time = next; 		
		keyFramesRot.push_back(key);
		time = next;
	}

	flags = NEXTKEY_RIGHT | NEXTKEY_POS;
	time = scene->interval.Start() - 1;
	while(c->GetNextKeyTime(time, flags, next)) {
		// looped?
		if(keyFramesTrans.size() && next == keyFramesTrans[0]->time)
			break;

		key = new ms3d_keyframe_t;
		key->time = next; 		
		keyFramesTrans.push_back(key);
		time = next;
	}
	return (keyFramesRot.size() || keyFramesTrans.size());
}

// Recursive function that traverses through the subanims of an Animatable
// Collects keyframes for a bone (joint).
void CJoint::GetKeyframes(Animatable *obj)
{
	Animatable *sub;
	int i;

	enum actionEnum {
		GET_NONE = 0,
		GET_EULER = 1,
		GET_POS = 2,
		numActionEnum
	};
	static actionEnum action;
	static int index;

	for(i = 0; i < obj->NumSubs(); i++) {
		sub = obj->SubAnim(i);
		if(!sub)
			continue;
		
		if(sub->ClassID() == Class_ID(HYBRIDINTERP_FLOAT_CLASS_ID , 0)) {
						
			IKeyControl *ikc = (IKeyControl*)sub->GetInterface(I_KEYCONTROL);
			IBezFloatKey floatKey;
			vector<ms3d_keyframe_t *> *vec; // Used to point to the correct vector of keyframes (rotation or translation).

			if (!ikc) {
				MessageBox(NULL, "Invalid keyframe interface for float controller.", "Error", MB_OK|MB_ICONERROR);
				return; // No interface available to access the keys...
			}
			if(index > 2) { //DEBUG: should NEVER happen!
				MessageBox(NULL, "Index > 2, Aborting... (BUG)", "Error", MB_OK|MB_ICONERROR);
				return; 
			}


			// Add to the correct vector of keys depending on type.
			if(action == GET_EULER)
				vec = &keyFramesRot;	
			else if(action == GET_POS)
				vec = &keyFramesTrans;	
			else // continue with next set. Happens for scale values (which arent used).
				return;


			float old = 0; 
			for(int k = 0; k < ikc->GetNumKeys(); k++) { // Go through all keys for this single axis/element controller.
				ikc->GetKey(k, &floatKey);
				ms3d_keyframe_t *key = NULL;
				// See if key with this timestamp already has been added (but with another index). 
				// If that is the case, Merge with the old.
				for(int j = 0; j < (int)vec->size(); j++) {
					key = (*vec)[j];
					if(key->time == (float)floatKey.time) 
						break; // use this old key.
					key = NULL;
				}
				if(!key) { // Create new.
					key = new ms3d_keyframe_t;				
					key->time = (float)floatKey.time;
					memset(key->mod, 0, sizeof(float) * 3);
					vec->push_back(key); // add to vector.	 
				}
				old += floatKey.val; // since each keyframe is relative to the previous.
				key->mod[index] = old;
			}
			index ++;
		}
		else { // Recurse.
			index = 0;
			if(sub->ClassID() == Class_ID(EULER_CONTROL_CLASS_ID , 0)) // rotation keyframes ahead.
				action = GET_EULER;
			else if(sub->ClassID() == Class_ID(IPOS_CONTROL_CLASS_ID)) // translation keys.
				action = GET_POS;
			else
				action = GET_NONE; // just traverse down until we get something useful.

			GetKeyframes(sub); // continue traversing down the sub-anims.
		}
	}
}

