//---------------------------------------------------------------------------
// FLModelCloner - Freelancer Model Cloner
// FLModelCloner.cpp - Main form code
// Copyright 2003-2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of FLModelCloner.
//
// FLModelCloner is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// FLModelCloner is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include <registry.hpp>

#include "Utils.h"
#include "TUTFFile.h"

#include "AboutBoxUnit.h"
#include "MainUnit.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TMainForm *MainForm;
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// ~TMainForm()
//
// Main form constructor
//
// - Allocate a new TUTFFile for the model
//---------------------------------------------------------------------------
__fastcall TMainForm::TMainForm(TComponent* Owner)
        : TForm(Owner)
{
  Model = new TUTFFile;
  VMeshCRCList = new THashedStringList;
  VMeshRefList = new THashedStringList;
  MaterialCRCList = new THashedStringList;
  RootVMesh = NULL;
  RootVMeshRef = NULL;
  RootMaterial = NULL;
  VMeshRefCount = 0;
  MaterialRefCount = 0;
}  // __fastcall TMainForm::TMainForm(TComponent* Owner)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ~TMainForm()
//
// Main form destructor
//---------------------------------------------------------------------------
__fastcall TMainForm::~TMainForm(void)
{
  delete Model;
  ClearModel();
  delete VMeshCRCList;
  delete VMeshRefList;
  delete MaterialCRCList;
}  // __fastcall TMainForm::~TMainForm(void)
//---------------------------------------------------------------------------

void __fastcall TMainForm::FormShow(TObject *Sender)
{
  BinDir = ExtractFileDir(_argv[0]);
  Application->HelpFile = BinDir + "\\FLModelCloner.hlp";
  IniFile = new TMemIniFile(BinDir + "\\FLModelCloner.ini");
  FLDir = IniFile->ReadString("Configuration", "FreelancerDir", "");
  if (FLDir.IsEmpty()) {                         // Check to see if we have an ini file
    TRegistry *Registry;
    Registry = new TRegistry();

    Registry->RootKey = HKEY_LOCAL_MACHINE;
    if (Registry->OpenKey("SOFTWARE\\Microsoft\\Microsoft Games\\Freelancer\\1.0",false))
      FLDir = Registry->ReadString("AppPath");
    delete Registry;
    if (FLDir.IsEmpty())
      FLDir = BinDir;
    IniFile->WriteString("Configuration", "FreelancerDir", FLDir);
  }  // if (FLDir.IsEmpty()) {
  WorkingDir = IniFile->ReadString("Configuration", "WorkingDir", FLDir);
  IniFile->WriteString("Configuration", "WorkingDir", WorkingDir);
  MainForm->Top = IniFile->ReadInteger("Configuration", "WindowTop", MainForm->Top);
  IniFile->WriteInteger("Configuration", "WindowTop", MainForm->Top);
  MainForm->Left = IniFile->ReadInteger("Configuration", "WindowLeft", MainForm->Left);
  IniFile->WriteInteger("Configuration", "WindowLeft", MainForm->Left);
  IniFile->UpdateFile();
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// FormActivate()
//
// Handler for the MainForm OnActivate event
//---------------------------------------------------------------------------
void __fastcall TMainForm::FormActivate(TObject *Sender)
{
  static bool FirstActivate = true;

  if (FirstActivate)
    FormActivateFirst(Sender);
}  // void __fastcall TMainForm::FormActivate(TObject *Sender)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// FormActivateFirst()
//
// Called by FormActivate() the first time the MainForm OnActivate event
// fires
//---------------------------------------------------------------------------
void __fastcall TMainForm::FormActivateFirst(TObject *Sender)
{
  ConfigTimer = new TTimer(MainForm);
  ConfigTimer->OnTimer = ConfigTimerFire;
  ConfigTimer->Interval = 500;                   // Wait half a second before firing the config timer
  ConfigTimer->Enabled  = true;
}  // void __fastcall TMainForm::FormActivateFirst(TObject *Sender)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ConfigTimerFire()
//
// There is no event that happens after a form is all nicely displayed and
// and painted and whatnot.  So we use a timer that is started when the
// form is first activated to fire this event that we then use to do
// anything that we want to have happen after the main form is displayed.
//---------------------------------------------------------------------------
void __fastcall TMainForm::ConfigTimerFire(TObject *Sender)
{

  delete ConfigTimer;                            // One shot timer, so shut delete it

}  // void __fastcall TMainForm::ConfigTimerFire(TObject *Sender)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// FormClose()
//
// MainForm OnClose event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::FormClose(TObject *Sender, TCloseAction &Action)
{
  // Save stuff to our ini file
  IniFile->WriteString("Configuration", "WorkingDir", WorkingDir);
  IniFile->WriteInteger("Configuration", "WindowTop", MainForm->Top);
  IniFile->WriteInteger("Configuration", "WindowLeft", MainForm->Left);
  IniFile->UpdateFile();
}  // void __fastcall TMainForm::FormClose(TObject *Sender, TCloseAction &Action)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuFile_OpenModelClick()
//
// Main menu->File->Open Model event handler
// Load a Freelancer UTF model into memory and scan it for meshes and
// materials.
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuFile_OpenModelClick(TObject *Sender)
{
  OpenDialog->InitialDir = WorkingDir;
  if (OpenDialog->Execute() && FileExists(OpenDialog->FileName)) {
    ModelFileName = OpenDialog->FileName;
    WorkingDir = ExtractFileDir(ModelFileName);
    if (Model->LoadFromFile(ModelFileName))
      ParseModel();
    UpdateInfo();
  }  //   if (OpenDialog->Execute() && FileExists(OpenDialog->FileName))
}  // void __fastcall TMainForm::menuFile_OpenModelClick(TObject *Sender)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuFile_ScanMatClick()
//
// Main menu->File->Scan .mat File event handler
// Load a Freelancer UTF material file into memory and scan it for material
// names that match the material CRC ID codes in the already loaded model.
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuFile_ScanMatClick(TObject *Sender)
{
  ScanDialog->InitialDir = WorkingDir;
  if (ScanDialog->Execute() && FileExists(ScanDialog->FileName)) {
    ScanMaterialFile(ScanDialog->FileName);
    UpdateInfo();
    treeIDListClick(Sender);
  }  // if (ScanDialog->Execute() && FileExists(ScanDialog->FileName))
}  // void __fastcall TMainForm::menuFile_ScanMatClick(TObject *Sender)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuFile_ExportNewModelClick()
//
// Main menu->File->Export New Model event handler
// Update the in-memory UTF file and save it.
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuFile_ExportNewModelClick(TObject *Sender)
{
  ExportDialog->InitialDir = WorkingDir;
  if (ExportDialog->Execute()) {
    AnsiString FileName = ExportDialog->FileName;
    CommitChanges();
    Model->SaveToFile(FileName);
    UpdateInfo();
    treeIDListClick(Sender);
  }  // if (ExportDialog->Execute())
}  // void __fastcall TMainForm::menuFile_ExportNewModelClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// menuFile_ExitClick()
//
// Main menu->File->Exit event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuFile_ExitClick(TObject *Sender)
{
  Close();
}  // void __fastcall TMainForm::menuFile_ExitClick(TObject *Sender)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuHelp_ContentsClick()
//
// Main menu->Help->Contents event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuHelp_ContentsClick(TObject *Sender)
{
  Application->HelpCommand(HELP_FINDER, 0);
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuHelp_HowToClick()
//
// Main menu->Help->How to clone a model event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuHelp_HowToClick(TObject *Sender)
{
  Application->HelpJump("Instructions");
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuHelp_OverviewClick
//
// Main menu->Help->Tecnhical Overview event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuHelp_OverviewClick(TObject *Sender)
{
  Application->HelpJump("Overview");
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuHelp_LicenseClick
//
// Main menu->Help->License event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuHelp_LicenseClick(TObject *Sender)
{
  Application->HelpJump("License");
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// menuHelp_AboutClick
//
// Main menu->Help->About event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::menuHelp_AboutClick(TObject *Sender)
{
  AboutForm->Execute();
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// treeIDListClick()
//
// treeIDList OnClick event handler
//---------------------------------------------------------------------------
void __fastcall TMainForm::treeIDListClick(TObject *Sender)
{
  TTreeNode *Node = treeIDList->Selected;
  if (!Node || Node->HasChildren) {              // Check to see if there is no selection, or if it's a root node
    edtOldName->Text = "";
    edtOldName->Enabled = false;
    lblOldCRC->Caption = "CRC = 0x00000000";
    lblOldCRC->Enabled = false;
    edtNewName->Text = "";
    edtNewName->Enabled = false;
    lblNewCRC->Caption = "CRC = 0x00000000";
    lblNewCRC->Enabled = false;
    lblOldName->Enabled = false;
    lblNewName->Enabled = false;
  } else {
    TCRCName *CRCName = (TCRCName *)Node->Data;
    edtOldName->Text = CRCName->OldName;
    edtOldName->Enabled = true;
    lblOldCRC->Caption = lblOldCRC->Caption.sprintf("CRC = 0x%8.8X", CRCName->OldCRC);
    lblOldCRC->Enabled = true;
    edtNewName->Text = CRCName->NewName;
    edtNewName->Enabled = true;
    lblNewCRC->Caption = lblNewCRC->Caption.sprintf("CRC = 0x%8.8X", CRCName->NewCRC);
    lblNewCRC->Enabled = true;
    lblOldName->Enabled = true;
    lblNewName->Enabled = true;
  } // if (!Node || Node->HasChildren)
}  // void __fastcall TMainForm::treeIDListClick(TObject *Sender)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// edtNewNameChange()
//
// edtNewName control OnChange event handler
// Calculate a new CRC, and update the TreeView icon for this item to
// represent whether it's been renamed or not.
//---------------------------------------------------------------------------
void __fastcall TMainForm::edtNewNameChange(TObject *Sender)
{
  TTreeNode *Node = treeIDList->Selected;
  TCRCName *CRCName;

  if (Node && !Node->HasChildren) {
    unsigned int CRC = generate_crc(edtNewName->Text.c_str());
    CRCName = (TCRCName *)Node->Data;

    lblNewCRC->Caption = lblNewCRC->Caption.sprintf("CRC = 0x%8.8X", CRC);
    CRCName->NewName = edtNewName->Text;
    CRCName->NewCRC = CRC;
    if (CRCName->OldCRC != CRC && CRCName->OldName != CRCName->NewName && !CRCName->NewName.IsEmpty()) {
      Node->ImageIndex = RENAMED;
      Node->SelectedIndex = RENAMED;
      Node->Text = CRCName->NewName;
    } else {
      Node->ImageIndex = NOTRENAMED;
      Node->SelectedIndex = NOTRENAMED;
      Node->Text = CRCName->OldName;
      CRCName->NewName = CRCName->OldName;
    }  // if (CRCName->OldCRC != CRC)
    UpdateInfo();
  }  //   if (Node)
}
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ParseModel()
//
// Go through a loaded UTF file containing a Freelancer Model and parse out
// any VMeshRef or model segments.
//
// Takes:   Nothing
// Returns: Nothing
// Throws:  Generic Exception if the VMeshData signature is incorrect
//---------------------------------------------------------------------------
void __fastcall TMainForm::ParseModel(void)
{

  ClearModel();
  VMeshCRCList->Clear();
  MaterialCRCList->Clear();

  // Add our root tree nodes
  RootVMesh = treeIDList->Items->Add(NULL, "VMesh IDs");
  RootVMesh->ImageIndex = VMESH;
  RootVMesh->SelectedIndex = VMESH;
  RootVMesh->StateIndex = -1;
  RootVMesh->Data = NULL;
  RootVMeshRef = treeIDList->Items->Add(NULL, "3DB (VMeshRef) Names");
  RootVMeshRef->ImageIndex = VMESHREF;
  RootVMeshRef->SelectedIndex = VMESHREF;
  RootVMeshRef->StateIndex = -1;
  RootVMeshRef->Data = NULL;
  RootMaterial = treeIDList->Items->Add(NULL, "Material IDs");
  RootMaterial->ImageIndex = MATERIAL;
  RootMaterial->SelectedIndex = MATERIAL;
  RootMaterial->StateIndex = -1;
  RootMaterial->Data = NULL;

  for (int n=0; n < Model->Count; n++) {

    TUTFTreeNode *Node = Model->Items[n];        // So we don't have to dereference this UTF node too many times

    if (Node->Name == "VMeshData") {             // Check to see if we have a VMesh file

      TTreeNode *VMeshNode, *MaterialNode;
      TVMeshHeader *VMeshHeader;
      AnsiString CRCHex;

      // Each VMesh name ends up being hashed to the VMeshID used in a VMeshRef
      // So the first thing we do here is add the VMesh name under the list of
      // VMesh IDs
      TCRCName *VMeshCRC = new TCRCName;
      VMeshCRC->OldName  = Model->Items[Node->Parent]->Name;
      VMeshCRC->OldCRC   = generate_crc(VMeshCRC->OldName.c_str());
      VMeshCRC->NewName  = VMeshCRC->OldName;
      VMeshCRC->NewCRC   = VMeshCRC->OldCRC;
      VMeshCRC->UTFNode  = Node->Parent;          // We need to know the UTF node for VMesh so we can change its name in the UTF file
      VMeshCRC->DataNode = 0;
      VMeshCRC->Identified = true;
      VMeshCRC->References = new TList;
      VMeshNode = treeIDList->Items->AddChildObject(RootVMesh, VMeshCRC->OldName, VMeshCRC);
      VMeshNode->ImageIndex = NOTRENAMED;
      VMeshNode->SelectedIndex = NOTRENAMED;
      VMeshNode->StateIndex = -1;
      VMeshCRCList->AddObject(IntToHex((int)generate_crc(VMeshCRC->OldName.c_str()),8), VMeshNode);

      // Now we parse through the actual VMeshData's data to get all the MaterialIDs
      VMeshHeader = (TVMeshHeader *)Node->Data;
      if (VMeshHeader->Signature1 != 1 || VMeshHeader->Signature2 != 4)
        throw Exception("Error parsing VMeshData node '" + VMeshCRC->OldName + "' - signature incorrect.");

      MaterialRefCount += VMeshHeader->MeshCount;
      for (int n=0; n < VMeshHeader->MeshCount; n++) {
        TVMeshMesh *VMesh = (TVMeshMesh *)((BYTE *)Node->Data + sizeof(TVMeshHeader) + sizeof(TVMeshMesh) * n);
        int HashIndex;

        CRCHex = IntToHex((int) VMesh->MaterialID, 8);
        HashIndex = MaterialCRCList->IndexOf(CRCHex);
        if (HashIndex == -1) {                   // Check if we have a MaterialID that we haven't added yet
          TCRCName *MaterialCRC = new TCRCName;
          MaterialCRC->OldName  = "ID=0x" + CRCHex;
          MaterialCRC->OldCRC   = VMesh->MaterialID;
          MaterialCRC->NewName  = MaterialCRC->OldName;
          MaterialCRC->NewCRC   = MaterialCRC->OldCRC;
          MaterialCRC->UTFNode  = 0;
          MaterialCRC->DataNode = 0;
          MaterialCRC->Identified = false;
          MaterialCRC->References = new TList;
          MaterialCRC->References->Add(&VMesh->MaterialID);
          MaterialNode = treeIDList->Items->AddChildObject(RootMaterial, MaterialCRC->OldName, MaterialCRC);
          MaterialNode->ImageIndex = NOTRENAMED;
          MaterialNode->SelectedIndex = NOTRENAMED;
          MaterialNode->StateIndex = -1;
          MaterialCRCList->AddObject(CRCHex, MaterialNode);
        } else {                                 // Add another reference to the existing list if we have this CRC already
          MaterialNode = (TTreeNode *)MaterialCRCList->Objects[HashIndex];
          ((TCRCName *)MaterialNode->Data)->References->Add(&VMesh->MaterialID);
        }  // if (MaterialCRCList->IndexOf(IntToHex(VMesh->MaterialID,8)) != -1)
      }  // for (int n=0; n < VMeshHeader->MeshCount; n++)
    }  // if (Model->Items[n]->Name == "VMeshData")
    else if (Model->Items[n]->Name == "VMeshWire") {
      TCRCName  *VMeshRefCRC = new TCRCName;
      TTreeNode *VMeshRefNode;

      VMeshRefCRC->OldName  = Model->Items[Node->Parent]->Name;
      VMeshRefCRC->OldCRC   = generate_crc(VMeshRefCRC->OldName.c_str());
      VMeshRefCRC->NewName  = VMeshRefCRC->OldName;
      VMeshRefCRC->NewCRC   = VMeshRefCRC->OldCRC;
      VMeshRefCRC->UTFNode  = Node->Parent;          // We need to know the UTF node for VMesh so we can change its name in the UTF file
      VMeshRefCRC->DataNode = 0;
      VMeshRefCRC->Identified = true;
      VMeshRefCRC->References = new TList;

      VMeshRefNode = treeIDList->Items->AddChildObject(RootVMeshRef, VMeshRefCRC->OldName, VMeshRefCRC);
      VMeshRefNode->ImageIndex = NOTRENAMED;
      VMeshRefNode->SelectedIndex = NOTRENAMED;
      VMeshRefNode->StateIndex = -1;
      VMeshRefList->AddObject(VMeshRefCRC->OldName, VMeshRefNode);

    }  // else if (Model->Items[n]->Name == "VMeshWire")
  }  // for (int n=0; n < Model->Count; n++)

  for (int n=0; n < Model->Count; n++) {

    TUTFTreeNode *Node = Model->Items[n];        // So we don't have to dereference this UTF node too many times

    // Is this a VMeshRef attribute - these have MeshID references in them
    if (Model->Items[n]->Name == "VMeshRef") {
      TVMeshRef *VMeshRef = (TVMeshRef *)Node->Data;
      AnsiString CRCHex;
      int HashIndex;

      CRCHex = IntToHex((int)VMeshRef->VMeshID,8);
      HashIndex = VMeshCRCList->IndexOf(CRCHex);
      if (HashIndex != -1) {
        TTreeNode *VMeshNode = (TTreeNode *)VMeshCRCList->Objects[HashIndex];
        ((TCRCName *)VMeshNode->Data)->References->Add(&VMeshRef->VMeshID);
        VMeshRefCount++;
      }  // if (HashIndex != -1)
    } // if (Model->Items[n]->Name == "VMeshRef")
    // Is this a .3db node's File Name attribute?
    else if (Model->Items[n]->Name == "File name") {
      // There shouldn't ever be a "File name" attribute that's not part of
      // of a Cmpnd node in a model file, but just in case...
      if (Model->Items[Model->Items[Model->Items[n]->Parent]->Parent]->Name == "Cmpnd") {
        int HashIndex;
        AnsiString TestData;

        HashIndex = VMeshRefList->IndexOf((char *)Model->Items[n]->Data);
        if (HashIndex != -1) {
          TTreeNode *VMeshRefNode = (TTreeNode *)VMeshRefList->Objects[HashIndex];
          ((TCRCName *)VMeshRefNode->Data)->DataNode = n;
        }  // if (HashIndex != -1)
      }  // if (Model->Items[Model->Items->Parent]->Name = "Cmpnd")
    }  // else if (Model->Items[n]->Name == "File name")
  }  // for (int n=0; n < Model->Count; n++)

} // void __fastcall TMainform::ParseModel(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ClearModel()
//
// Clean up a model's data
//
// Takes:   Nothing
// Returns: Nothing
// Throws:  Dunno yet
//---------------------------------------------------------------------------
void __fastcall TMainForm::ClearModel(void)
{
  for (int n=0; n<treeIDList->Items->Count; n++) {
    TCRCName *CRCName = (TCRCName *)treeIDList->Items->Item[n]->Data;

    if (CRCName) {
      delete CRCName->References;
      delete CRCName;
    }  // if (CRCName)
  }  // for (int n=0; n<treeIDList->Items->Count; n++)

/*
  if (RootVMesh)
    if (RootVMesh->Count)
      for (int n=0; n<RootVMesh->Count; n++) {
        TCRCName *VMeshCRC = (TCRCName *)RootVMesh->Item[n]->Data;
        delete VMeshCRC->References;
        delete VMeshCRC;
      }  // for (int n=0; n<RootVMesh->Count; n++)
  if (RootMaterial)
    if (RootMaterial->Count)
      for (int n=0; n<RootMaterial->Count; n++) {
        TCRCName *MaterialCRC = (TCRCName *)RootMaterial->Item[n]->Data;
        delete MaterialCRC->References;
        delete MaterialCRC;
      }  // for (int n=0; n<RootVMesh->Count; n++)
*/

  RootVMesh = NULL;
  RootVMeshRef = NULL;
  RootMaterial = NULL;
  VMeshRefCount = 0;
  MaterialRefCount = 0;
  treeIDList->Items->Clear();
  VMeshCRCList->Clear();
  VMeshRefList->Clear();
  MaterialCRCList->Clear();
  
}  // void __fastcall TMainForm::ClearModel(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// ScanMaterialFile()
//
// Load a Freelancer UTF material library file and scan it for material names
// that match the matericl CRC ID codes in an already loaded model.  We are
// matching names to the IDs so that they can be renamed more intelligently.
//
// Takes: Ansistring filename
// Returns: Nothing
// Throws: Dunno yet
//---------------------------------------------------------------------------
void __fastcall TMainForm::ScanMaterialFile(AnsiString FileName)
{
  TUTFFile *MatLib = new TUTFFile;
  AnsiString CRCHex;
  int Index;

  if (!MatLib->LoadFromFile(FileName))
    throw Exception("File does not appear to be a UTF file.");
  for (int n=0; n<MatLib->Count; n++) {
    CRCHex = IntToHex((int)generate_crc(MatLib->Names[n].c_str()),8);
    Index = MaterialCRCList->IndexOf(CRCHex);
    if ( Index != -1) {
      TTreeNode *Node = (TTreeNode *)MaterialCRCList->Objects[Index];
      TCRCName *CRCName = (TCRCName *)Node->Data;
      Node->Text = MatLib->Names[n];
      CRCName->Identified = true;
      if (CRCName->OldName == CRCName->NewName)
        CRCName->NewName = MatLib->Names[n];
      CRCName->OldName = MatLib->Names[n];
    }  // if ( Index != -1)
  }  //  void __fastcall TMainForm::ScanMaterialFile(AnsiString FileName)
  delete MatLib;
}  // void __fastcall TMainForm::ScanMaterialFile(AnsiString FileName)

//---------------------------------------------------------------------------
// UpdateInfo()
//
// Update the model information & enable/disable Main Menu items
//---------------------------------------------------------------------------
void __fastcall TMainForm::UpdateInfo(void)
{
  int RenameCount, IdentifyCount;

  if (RootVMesh && RootMaterial) {

    lblVMeshInfo->Caption = "Model has " + String(VMeshCRCList->Count) + " VMeshes which are referenced " + String(VMeshRefCount) + " times.";
    RenameCount = 0;
    for (int n=0; n < RootVMesh->Count; n++) {
      if (RootVMesh->Item[n]->ImageIndex != RENAMED)
        RenameCount++;
    }  // for (int n=0; n < RootVMesh->Count; n++)
    lblVMeshRename->Caption = "There are " + String(RenameCount) + " VMeshes left to rename.";
    if (RenameCount)
      lblVMeshRename->Font->Color = clRed;
    else
      lblVMeshRename->Font->Color = clGreen;

    lblVMeshRefInfo->Caption = "Model has " + String(VMeshRefList->Count) + " *.3DB (VMeshRef) names.";
    RenameCount = 0;
    for (int n=0; n < RootVMeshRef->Count; n++) {
      if (RootVMeshRef->Item[n]->ImageIndex != RENAMED)
        RenameCount++;
    }  // for (int n=0; n < RootVMeshRef->Count; n++)
    lblVMeshRefRename->Caption = "There are " + String(RenameCount) + " 3DB names left to change.";
    if (RenameCount)
      lblVMeshRefRename->Font->Color = clRed;
    else
      lblVMeshRefRename->Font->Color = clGreen;

    lblMaterialInfo->Caption = "Model has " + String(MaterialCRCList->Count) + " materials which are referenced " + String(MaterialRefCount) + " times.";
    RenameCount = IdentifyCount = 0;
    for (int n=0; n < RootMaterial->Count; n++) {
      if (!((TCRCName *)RootMaterial->Item[n]->Data)->Identified)
        IdentifyCount++;
      if (RootMaterial->Item[n]->ImageIndex != RENAMED)
        RenameCount++;
    }  // for (int n=0; n < RootMaterial->Count; n++)
    lblMaterialIdentify->Caption = "There are " + String(IdentifyCount) + " materials left to identify.";
    if (IdentifyCount)
      lblMaterialIdentify->Font->Color = clRed;
    else
      lblMaterialIdentify->Font->Color = clGreen;
    lblMaterialRename->Caption = "There are " + String(RenameCount) + " materials left to rename.";
    if (RenameCount)
      lblMaterialRename->Font->Color = clYellow;
    else
      lblMaterialRename->Font->Color = clGreen;


    lblVMeshInfo->Visible = true;
    lblVMeshRename->Visible = true;
    lblVMeshRefInfo->Visible = true;
    lblVMeshRefRename->Visible = true;
    lblMaterialInfo->Visible = true;
    lblMaterialIdentify->Visible = true;
    lblMaterialRename->Visible = true;
    menuFile_ScanMat->Enabled = true;
    menuFile_ExportNewModel->Enabled = true;
  } else {
    lblVMeshInfo->Visible = false;
    lblVMeshRename->Visible = false;
    lblVMeshRefInfo->Visible = false;
    lblVMeshRefRename->Visible = false;
    lblMaterialInfo->Visible = false;
    lblMaterialIdentify->Visible = false;
    lblMaterialRename->Visible = false;
    menuFile_ScanMat->Enabled = false;
    menuFile_ExportNewModel->Enabled = false;
  }  // if (RootVMesh && RootMaterial)
}  // void __fastcall TMainForm::UpdateInfo(void)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// CommitChanges()
//
// Update the model in memory with all the new CRC values and node names.
//---------------------------------------------------------------------------
void __fastcall TMainForm::CommitChanges(void)
{
  for (int n=0; n<treeIDList->Items->Count; n++) {
    TCRCName *CRCName = (TCRCName *)treeIDList->Items->Item[n]->Data;

    if (CRCName) {
      if (CRCName->UTFNode)
        Model->Names[CRCName->UTFNode] = CRCName->NewName;
      if (CRCName->DataNode)
        Model->SetData(CRCName->DataNode, CRCName->NewName);
      if (CRCName->References->Count)
        for (int x=0; x<CRCName->References->Count; x++)
          *(unsigned int *)CRCName->References->Items[x] = CRCName->NewCRC;
      CRCName->OldName = CRCName->NewName;
      CRCName->OldCRC  = CRCName->NewCRC;
      treeIDList->Items->Item[n]->ImageIndex = NOTRENAMED;
      treeIDList->Items->Item[n]->SelectedIndex = NOTRENAMED;
    }  // if (!treeIDList->Items->Item[n]->HasChildren)
  }  // for (int n=0; n<treeIDList->Items->Count; n++)
}  // void __fastcall TMainForm::CommitChanges(void)
//---------------------------------------------------------------------------

