//---------------------------------------------------------------------------
// FLModelCloner - Freelancer Model Cloner
// FLModelCloner.cpp - TUTFFile - UTF file manipulation class
// Copyright 2003-2005, Kurt Fitzner <kfitzner@excelcia.org>
//---------------------------------------------------------------------------
// This file is part of FLModelCloner.
//
// FLModelCloner is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// FLModelCloner is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include <stdio.h>

#include "TUTFFile.h"


//---------------------------------------------------------------------------
// TUTFFile()
//
// Default Constructor
//
// Takes:   Nothing
// Returns: It's a constructor, dummy
// Throws:  bad_alloc if it can't make a new TList
//---------------------------------------------------------------------------
__fastcall TUTFFile::TUTFFile(void) : System::TObject()
{
  Nodes = new TList;
}  // __fastcall TUTFFile::TUTFFile(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// ~TUTFFile()
//
// Default destructor
//
// Takes:   Nothing
// Returns: It's a destructor, dummy
// Throws:  Nothing but tantrums
//---------------------------------------------------------------------------
__fastcall TUTFFile::~TUTFFile(void)
{
  TUTFTreeNode *node;

  Clear();
  delete Nodes;
}  // __fastcall TUTFFile::TUTFFile(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Clear()
//
// Reset the object to nil - free up all the wasted... er used memory
//
// Takes:   Nothing
// Returns: Nothing
// Throws:  Nothing but tantrums
//---------------------------------------------------------------------------
void __fastcall TUTFFile::Clear(void)
{
  TUTFTreeNode *node;

  if (Nodes)
    for (int n = 0; n < Nodes->Count; n++) {
      node = (TUTFTreeNode *)Nodes->Items[n];
      if (node->Data)
        free(node->Data);
        delete node;
    }  // while (Nodes->Count)
    Nodes->Clear();
}  // void __fastcall TUTFFile::Clear(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// LoadFromFile()
//
// Loads a UTF file into memory from a disk file.
//
// Takes: Ansistring file name
// Returns: bool true if loads ok, false if the file isn't a UTF file.
// Throws: EFOpenError if it can't fopen the file
//         EReadError if it can't read the file
//         bad_alloc if it can't allocate memory
//---------------------------------------------------------------------------
bool __fastcall TUTFFile::LoadFromFile(AnsiString &FileName)
{
  int             UTFFile;
  long int        size, read;
  BYTE           *buffer = NULL;
  TUTFFileHeader *Header;
  int            *Parents = NULL;

  UTFFile = FileOpen(FileName.c_str(), fmOpenRead);
  if (UTFFile == -1)
    throw EFOpenError("Could not open UTF File: '" + String(strerror(errno)) + "'");

  size = FileSeek(UTFFile, 0, 2);
  FileSeek(UTFFile, 0, 0);
  buffer = (BYTE *) malloc(size);
  if (!buffer)
    OutOfMemoryError();

  try {

    read = FileRead(UTFFile, buffer, size);
    if ( read !=  size)
      throw EReadError("Could not read UTF file: '" + String(strerror(errno)) + "'");
    FileClose(UTFFile);
    Header = (TUTFFileHeader *) buffer;
    if (Header->Signature1 != 0x20465455 || Header->Signature2 != 0x0101) {
      delete buffer;
      return(false);
    }  // if (Header->Signature1 != 0x20465455 || Header->Signature2 != 0x0101)
    Clear();
    Parents = (int *) calloc(Header->TreeSegmentSize, sizeof(int));
    if (!Parents)
      OutOfMemoryError();

    for (unsigned n = 0; n < Header->TreeSegmentSize / 44; n++) {

      TUTFFileTreeNode *FileNode = (TUTFFileTreeNode *)(buffer + Header->TreeSegmentOffset + n * 44);
      TUTFTreeNode *node = new TUTFTreeNode;

      node->Name = String((char *) buffer + Header->StringSegmentOffset + FileNode->StringOffset);
      node->IsLeaf = FileNode->Flags & 0x80;
      node->Child = (node->IsLeaf)?0:FileNode->ChildOffset / 44;
      Parents[node->Child] = n;
      node->Sibling = FileNode->SiblingOffset / 44;
      node->Parent = Parents[n];
      if (node->Parent && node->Sibling) {
        TUTFTreeNode *sibling = node;

        Parents[node->Sibling] = node->Parent;
        // Some malformed UTF files had siblings in the wrong order in the file.
        // The protocol allows it (or seems to), but it's stupid to write it this way.
        while (sibling->Sibling < n) {
          sibling = (TUTFTreeNode *)Nodes->Items[sibling->Sibling];
          sibling->Parent = node->Parent;
          Parents[sibling->Sibling] = n;
        }  // while (sibling->Sibling < n)
      }  // if (node->Parent & node->Sibling)

      // In the case (which will probably never happen) that a node's child
      // was read in before its parent, we check and set the parent attribute
      // on the child.
      if (node->Child && node->Child < n)
        ((TUTFTreeNode *)Nodes->Items[node->Child])->Parent = n;
      node->AllocatedDataSize = FileNode->AllocatedSize;
      node->DataSize = FileNode->Size1;
      if (node->AllocatedDataSize) {
        node->Data = malloc(node->AllocatedDataSize);
        if (!node->Data)
          OutOfMemoryError();
        memcpy(node->Data, buffer + Header->DataSegmentOffset + FileNode->ChildOffset, node->DataSize);
      } else
        node->Data = NULL;
      node->Unknown = FileNode->Unknown;
      node->Time1   = FileNode->Time1;
      node->Time2   = FileNode->Time2;
      node->Time3   = FileNode->Time3;
      Nodes->Add(node);

    }  // for (int n = 0; n < Header->TreeSegmentSize / 44; n++)

    free(buffer);
    free(Parents);

  }  // try

  catch(Exception &e) {
    if (buffer)
      free(buffer);                         // Free up the mem if we fail
    if (Parents)
      free(Parents);
    while (Nodes->Count > 0) {
      TUTFTreeNode *node = (TUTFTreeNode *)Nodes->Items[0];
      if (node->Data)
        free(node->Data);
      delete node;
      Nodes->Delete(0);
    }  // while (Nodes->Count > 0)
    throw Exception(e);
  }  // catch

  return(true);
}  // bool __fastcall Open(AnsiString &FileName)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// SaveToFile()
//
// Saves a UTF file from memory to a disk file.
//
// Takes: Ansistring file name
// Returns: bool true if saves ok, false if not
// Throws: EFOpenError if it can't fopen the file
//         EWriteError if it can't read the file
//         bad_alloc if it can't allocate memory
//---------------------------------------------------------------------------
bool __fastcall TUTFFile::SaveToFile(AnsiString &FileName)
{
  if (Nodes && Nodes->Count) {
    TUTFFileHeader FileHeader;
    int UTFFile;
    int StringOffset, DataOffset;
    const int Zero = 0;

    try {
//      UTFFile = FileOpen(FileName, fmCreate);
      UTFFile = FileCreate(FileName);
      if (UTFFile == -1)
        throw EFOpenError("Could not open file for writing:\r\n" + String(strerror(errno)));
      FileSeek(UTFFile, 0, 0);
      if (!SetEndOfFile((HANDLE)UTFFile))
        throw EWriteError("Could not write to file while truncating:\r\n" + String(strerror(errno)));

      // Calculate sizes and make a header
      FileHeader.Signature1 = 0x20465455;
      FileHeader.Signature2 = 0x0101;
      FileHeader.TreeSegmentOffset = 56;
      FileHeader.TreeSegmentSize = Nodes->Count * sizeof(TUTFFileTreeNode);
      FileHeader.HeaderOffset = 0;
      FileHeader.HeaderSize = 44;
      FileHeader.StringSegmentOffset = FileHeader.TreeSegmentOffset + FileHeader.TreeSegmentSize;
      FileHeader.StringSegmentAllocatedSize = 1;
      for (int n=0; n<Nodes->Count; n++)
        FileHeader.StringSegmentAllocatedSize += ((TUTFTreeNode *)Nodes->Items[n])->Name.Length() + 1;
      FileHeader.StringSegmentSize = FileHeader.StringSegmentAllocatedSize;
      FileHeader.DataSegmentOffset = FileHeader.StringSegmentOffset + FileHeader.StringSegmentSize;
      FileHeader.Unknown = 0;

      // Write the header
      if (FileWrite(UTFFile, &FileHeader, sizeof(TUTFFileHeader)) != sizeof(TUTFFileHeader))
        throw EWriteError("Could not write to file while writing header:\r\n" + String(strerror(errno)));

      // Write twelve bytes of zeros for filler
      FileWrite(UTFFile, &Zero, 4);
      FileWrite(UTFFile, &Zero, 4);
      FileWrite(UTFFile, &Zero, 4);

      // Write TreeSegment
      StringOffset = 1;
      DataOffset = 0;
      for (int n=0; n<Nodes->Count; n++) {
        TUTFFileTreeNode  FileNode;
        TUTFTreeNode     *Node = (TUTFTreeNode *)Nodes->Items[n];

        // Set up the node
        FileNode.SiblingOffset = Node->Sibling * sizeof(TUTFFileTreeNode);
        FileNode.StringOffset = StringOffset;
        FileNode.Flags = Node->IsLeaf?0x80:0x10;
        FileNode.Unknown = Node->Unknown;
        FileNode.ChildOffset = Node->IsLeaf?DataOffset:Node->Child*sizeof(TUTFFileTreeNode);
        FileNode.AllocatedSize = Node->AllocatedDataSize;
        FileNode.Size1 = FileNode.Size2 = Node->DataSize;
        FileNode.Time1 = Node->Time1;
        FileNode.Time2 = Node->Time2;
        FileNode.Time3 = Node->Time3;

        // Write the node
        if (FileWrite(UTFFile, &FileNode, sizeof(TUTFFileTreeNode)) != sizeof(TUTFFileTreeNode))
          throw EWriteError("Could not write to file while writing node:\r\n" + String(strerror(errno)));

        // Increment segment offsets
        StringOffset += Node->Name.Length() + 1;
        DataOffset += Node->AllocatedDataSize;
      }  // for (int n=0; n<Nodes->Count; n++)

      // Write the string segment
      // String segments always have a leading null byte - dunno why
      FileWrite(UTFFile, &Zero, 1);
      for (int n=0; n<Nodes->Count; n++) {
        TUTFTreeNode     *Node = (TUTFTreeNode *)Nodes->Items[n];

        FileWrite(UTFFile, Node->Name.c_str(), Node->Name.Length());
        FileWrite(UTFFile, &Zero, 1);
      }  // for (int n=0; n<Nodes->Count; n++)

      // Write the data segment
      for (int n=0; n<Nodes->Count; n++) {
        TUTFTreeNode     *Node = (TUTFTreeNode *)Nodes->Items[n];

        if (Node->IsLeaf && Node->AllocatedDataSize)
          FileWrite(UTFFile, Node->Data, Node->AllocatedDataSize);
      }  // for (int n=0; n<Nodes->Count; n++)

      // FileClose(UTFFile);

    }  // try
    __finally {
      if (UTFFile != -1)
        FileClose(UTFFile);
    }  // __finally
    return true;
  } else {
    return false;
  }  // if (Nodes && Nodes->Count)


}  // bool __fastcall TUTFFile::SaveToFile(AnsiString &FileName)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// SetData(int Index, AnsiString NewData)
//
// Set a UTF node's data to be the given string, freeing previous data
// if any.
//
// Takes: Integer index indicating which node, and an AnsiString containing
//        the data
// Returns: Nothing
// Throws: EOutOfMemory if it can't alloc enough space
//---------------------------------------------------------------------------
void __fastcall TUTFFile::SetData(int Index, AnsiString NewData)
{
  TUTFTreeNode *Node;
  void         *Data;
  int           DataSize;
  const int     Length = NewData.Length() + 1;

  if (Index > Nodes->Count)
    return;

  DataSize = (Length%4)?(Length/4+1)*4:Length;
  Data = calloc(DataSize, 1);
  if (!Data)
    OutOfMemoryError();
  memcpy(Data, NewData.c_str(), Length);
  Node = (TUTFTreeNode *)Nodes->Items[Index];
  if (Node->Data)
    free(Node->Data);
  Node->Data = Data;

}  // void __fastcall TUTFFile::SetData(int Index, AnsiString NewData)
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// GetCount()
//
// Read access method for TUTFFile::Count property
//---------------------------------------------------------------------------
int __fastcall TUTFFile::GetCount(void)
{
  if (Nodes)
    return(Nodes->Count);
  else
    return(0);
}  // int __fastcall TUTFFile::GetCount(void)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// GetItems()
//
// Read access method for TUTFFile::Items[] property
//---------------------------------------------------------------------------
TUTFTreeNode * __fastcall TUTFFile::GetItems(int Index)
{
  if (Nodes)
    return((TUTFTreeNode *)Nodes->Items[Index]);
  else
    return(NULL);
}  // int __fastcall TUTFFile::GetItems(int Index)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// GetNames()
//
// Read access method for TUTFFile::Names[] property
//---------------------------------------------------------------------------
AnsiString __fastcall TUTFFile::GetNames(int Index)
{
  if (Nodes)
    return(((TUTFTreeNode *)Nodes->Items[Index])->Name);
  else
    return("");
}  // int __fastcall TUTFFile::GetNames(int Index)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// PutNames()
//
// Write access method for TUTFFile::Names[] property
//---------------------------------------------------------------------------
void __fastcall TUTFFile::PutNames(int Index, AnsiString Name)
{
  if (Nodes)
    ((TUTFTreeNode *)Nodes->Items[Index])->Name = Name;
}  // int __fastcall TUTFFile::PutNames(AnsiString Name, int Index)
//---------------------------------------------------------------------------


#pragma package(smart_init)

