using System;
using System.IO;

namespace TLR.Freelancer.Utility
{

	/// <summary>
	/// Breaks a Freelancer INI file down into "chunks" that
	/// represent functional tokens: headers, data elements such as
	/// value names and values, and delimiters such as commas and
	/// equal signs.
	/// All data is appended to a Queue instance allowing first-in
	/// first-out parsing.
	/// This chunker does NO assessment whatsoever about meaning or
	/// validity of the data; it is not intended to do so. It DOES
	/// correctly interpret quoted strings. 
	/// </summary>
	public class IniChunker
	{

		// The following are numeric codes for significant special characters.
		private const int OpenBrace = 0x5b; //[
		private const int CloseBrace = 0x5d; //]
		private const int EqualSign = 0x3d; //=
		private const int Semicolon = 0x3b; //;
		private const int Doublequote = 0x22; //;
		private const int Comma = 0x2c; //,
		private const int Space = 0x20;
		private const int Tab = 0x09;
		private const int Linefeed = 0x0a;
		private const int CarriageReturn = 0x0d;
		private int streamState = 0;
		private System.Collections.Queue chunkQueue;

		/// <summary>
		/// Parses a Freelancer INI file; file MUST be text, not BINI-compressed.
		/// </summary>
		/// <param name="filepath">Path to the file to be parsed.</param>
		public void Read(string filepath)
		{
			chunkQueue = new System.Collections.Queue();
			StreamReader sr = new StreamReader(filepath);
			System.Text.StringBuilder token;
			while (sr.Peek() != -1 )
			{
				int datum = sr.Peek();
				switch (streamState)
				{
						// Based on current stream "state" we choose how
						// to parse characters.
						// This is a very crude parser. After we enter
						// a specific state - in a header, string, value, etc -
						// we stay in the case statement consuming characters
						// until it terminates.
					case (int)StreamStates.Ground:
						if ((datum == Tab) | (datum == Space))
						{ // We skip past tabs and spaces
							sr.Read();
						}
						else if ((datum == 0x0a) | (datum == 0x0d))
						{ // line terminations make us emit 1 and "eat" any others
							EnqueueChunk((char)0x0a);
							datum = sr.Peek();
							while ( (datum != -1 ) &&
								((datum == 0x0d) | (datum == 0x0a)) )
							{
								sr.Read(); datum = sr.Peek();
              }
						}
						else if (datum == OpenBrace)
						{ streamState = (int)StreamStates.Header;}
							// non-whitespace means we are entering a token
						else if (datum == Doublequote)
						{ streamState = (int)StreamStates.Comment;}
						else if (datum == Semicolon)
						{ streamState = (int)StreamStates.Comment;}
						else if ((datum == Comma)
							|(datum == EqualSign))
						{ EnqueueChunk((char)sr.Read());}
						else
						{ // consume stuff here during construction
							streamState = (int)StreamStates.Value;
						}

						continue; // end of things we do in whitespace...

					case (int)StreamStates.Header:
						token = new System.Text.StringBuilder();
						while ((sr.Peek() != CloseBrace)
							&& (sr.Peek() != 0x0a)
							&& (sr.Peek() != 0x0d)
							&& (sr.Peek() != -1))
							{token.Append((char)sr.Read());}
						if (sr.Peek() == CloseBrace){token.Append((char)sr.Read());}
							EnqueueChunk(token);
							streamState = (int)StreamStates.Ground;
						continue;

					case (int)StreamStates.Value:
						token = new System.Text.StringBuilder();
						while (
							(sr.Peek() != Comma)
							&& (sr.Peek() != EqualSign)
							&& (sr.Peek() != 0x0a)
							&& (sr.Peek() != 0x0d)
							&& (sr.Peek() != 0x09)
							&& (sr.Peek() != 0x20)
							&& (sr.Peek() != -1))
						{token.Append((char)sr.Read());}
						EnqueueChunk(token);
						streamState = (int)StreamStates.Ground;
						continue;

					case (int)StreamStates.Quoted:
						token = new System.Text.StringBuilder();
						while ( (sr.Peek() != Doublequote)
							&& (sr.Peek() != 0x0a)
							&& (sr.Peek() != 0x0d)
							&& (sr.Peek() != -1))
						{token.Append((char)sr.Read());}
						if (sr.Peek() == Doublequote){token.Append((char)sr.Read());}
						EnqueueChunk(token);
						streamState = (int)StreamStates.Ground;
						continue;

					case (int)StreamStates.Comment:
						token = new System.Text.StringBuilder();
						while ((sr.Peek() != 0x0a)
							&& (sr.Peek() != 0x0d)
							&& (sr.Peek() != -1)){
							token.Append((char)sr.Read()); 
						}
						EnqueueChunk(token);
						streamState = (int)StreamStates.Ground;
						continue;
				}
			}
			return;
		}

		
		/// <summary>
		/// Enqueues a chunk of data.
		/// </summary>
		/// <param name="chunk"></param>
		private void EnqueueChunk(object chunk)
		{
			chunkQueue.Enqueue(chunk.ToString());
    }

		/// <summary>
		/// Stream parser states.
		/// </summary>
		private enum StreamStates
		{
			Ground = 0,
			Header = 1,
			Value = 2,
			Quoted = 3,
			Comment = 4
		}

		/// <summary>
		/// A queue containing parsed tokens from the INI file.
		/// </summary>
		public System.Collections.Queue TokenQueue { get { return chunkQueue; } }

	}
}
