// FLchar.cpp

#include <time.h>
#include "FLchar.h"

//----------------------------------------------------------------------------
// Constructor - load character from file

FLchar::FLchar(char * directory, char * filename, std::string sort)
: dir(directory), file(filename), sortKey(sort), 
  name(""), description(""), group(""), rank(0), money(0), kills(0),
  missions(0), failed(0), timeOnline(0), timeLogin(0), systems(0), bases(0), holes(0)
{
    FILE * input;
    bool encoded = false;

    // Initialise file index, and open file
    long findex = 0;
    input = fopen(filename,"r");

    // If open failed, return 
    if (input == NULL) {
        return;
    }

    // Gobble up first four bytes: FLS1
    char buffer[5];
    fread(&buffer, 4, 1, input);

    if (!strncmp(buffer,"FLS1", 4)) {
      encoded = true;
    } else {
      fseek(input, 0, SEEK_SET);
    }

    // For each line in file...
    while (!feof(input)) {
        // Get line as a list of words
        varList line = getFLline(input, encoded, findex);
        if (line.size() < 2)
            continue;

        // Store desired parameters in member variables
        if (line[0] == "name") {
            name = getTextFromHex(line[1]);
        } else if (line[0] == "description") {
            description = getTextFromHex(line[1]);
            timeLogin = getTimeFromText(description);
        } else if (line[0] == "rep_group") {
            group = convertGroup(line[1]);
        } else if (line[0] == "rank") {
            rank = atoi(line[1].c_str());
        } else if (line[0] == "money") {
            money = atoi(line[1].c_str());
        } else if (line[0] == "rm_completed") {
            missions += atoi(line[2].c_str());
        } else if (line[0] == "ship_type_killed") {
            kills += atoi(line[2].c_str());
        } else if (line[0] == "sys_visited") {
            systems++;
        } else if (line[0] == "base_visited") {
            bases++;
        } else if (line[0] == "holes_visited") {
            holes++;
        } else if (line[0] == "total_time_played") {
            timeOnline = atol(line[1].c_str());
        }
    }

    // Close file and return
    fclose(input);   
    return; 
}

//----------------------------------------------------------------------------
// Returns false if character has never logged in, 
// has a rank below minRank, or logged in more than maxDays ago

bool FLchar::Active(const FLoptions &options)
{	
    if (description == "")
        return false;

    if (rank < options.minRank)
        return false;

    int dTime = time(NULL) - timeLogin;
    if (options.maxDays > 0 && dTime > options.maxDays * ONE_DAY)
        return false;

    return true;
}

// Sort function based on current sort key
bool FLchar::operator< (FLchar other) {
	if (sortKey == "Login") {
 		return (timeLogin > other.timeLogin);
	}
	else if (sortKey == "Time") {
 		return (timeOnline > other.timeOnline);
	}
	else if (sortKey == "Money") {
		return ((money > other.money) ||
    			(money == other.money && rank > other.rank) ||
    			(money == other.money && rank == other.rank && kills > other.kills));
	}
	else if (sortKey == "Kills") {
		return ((kills > other.kills) ||
    			(kills == other.kills && rank > other.rank) ||
    			(kills == other.kills && rank == other.rank && money > other.money));
	}
	else {
		return ((rank > other.rank) ||
    			(rank == other.rank && money > other.money) ||
    			(rank == other.rank && money == other.money && kills > other.kills));
	}
	return false;
}

//----------------------------------------------------------------------------
// Output all member variables to stdout

void FLchar::print(FILE *hOutput, int position, const FLoptions &options)
{
    char timeStr[100] = {0};
    char dateStr[100] = {0};

	// Cheat detection
    double suspect = (double) (rank - kills) / ((double) timeOnline / ONE_HOUR);
    long dTime = time(NULL) - timeLogin;

    // Output all members as entries in table row
    if (rank > 1 && suspect > options.cheatLevel) 
        fprintf(hOutput, "<TR CLASS=\"suspect\">\n");
    else if (dTime <= options.activeHours * ONE_HOUR)
        fprintf(hOutput, "<TR CLASS=\"active\">\n");
    else
        fprintf(hOutput, "<TR>\n");

    // Comment containing character account dir and file name
    if (options.withfile)
	  fprintf(hOutput, "\t<!-- File: %s/%s -->\n", dir.c_str(), file.c_str());

    // Position in ranking
    fprintf(hOutput, "\t<TD CLASS=\"pos\">%i</TD>\n", position);

    // Name in HTML safe UNICODE
    fprintf(hOutput, "\t<TD CLASS=\"name\">%s</TD>\n", name.c_str());

    // Group (commented out)
    fprintf(hOutput, "\t<!--TD CLASS=\"group\">%s</TD-->\n", group.c_str());

    // Rank
    fprintf(hOutput, "\t<TD CLASS=\"rank\">%i</TD>\n", rank);

    // Money, with comma-seperated thousands
	char withCommas[50];
	getCommaThousands(withCommas, money);
    fprintf(hOutput, "\t<TD CLASS=\"money\">$%s</TD>\n", withCommas);

    // Kills, with comma-seperated thousands
	getCommaThousands(withCommas, kills);
    fprintf(hOutput, "\t<TD CLASS=\"kills\">%s</TD>\n", withCommas);

    // Time spent online, in hours:minutes:seconds
    strftime(timeStr, 12, "%M:%S", localtime(&timeOnline));
    fprintf(hOutput, "\t<TD CLASS=\"time\">%i:%s</TD>\n", int(timeOnline / ONE_HOUR), timeStr);

	// Number of missions and systems, bases and holes visited
	fprintf(hOutput, "\t<TD CLASS=\"missions\">%i</TD>\n", missions);
	fprintf(hOutput, "\t<TD CLASS=\"systems\">%i</TD>\n", systems);
	fprintf(hOutput, "\t<TD CLASS=\"bases\">%i</TD>\n", bases);
	fprintf(hOutput, "\t<TD CLASS=\"holes\">%i</TD>\n", holes);

	// Last login time, as "HH:MM:SS - dd mmm yy"
	strftime(timeStr, 99, "%X", localtime(&timeLogin));

	if (dTime < time(NULL) % ONE_DAY)
		strcpy(dateStr, "Today");
	else if (dTime < time(NULL) % TWO_DAYS)
		strcpy(dateStr, "Yesterday");
	else
		strftime(dateStr, 99, "%d %b %y", localtime(&timeLogin));

	fprintf(hOutput, "\t<TD CLASS=\"login\">%s - %s</TD>\n", timeStr, dateStr);

	// End of table row
	fprintf(hOutput, "</TR>\n");
}

