<?php
// wcf imports
require_once(WCF_DIR.'lib/data/shoutbox/ShoutboxEntryEditor.class.php');
require_once(WCF_DIR.'lib/data/shoutbox/ShoutboxEntryList.class.php');
require_once(WCF_DIR.'lib/page/AbstractSecurePage.class.php');

/**
 * Starts the execution of actions on shoutbox entries.
 * 
 * @author	Sebastian ttl
 * @copyright	2009 Sebastian ttl
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @package	com.mysteryforce.wcf.shoutbox
 * @subpackage	lib.page
 * @category 	Shoutbox
 */
class ShoutboxActionPage extends AbstractSecurePage {
	public $entryID = 0;
	public $username = '';
	public $message = '';
	public $commandName = '';
	
	/**
	 * @see Page::readParameters()
	 */
	public function readParameters() {
		AbstractPage::readParameters(); // no security token validation here
		
		if (isset($_REQUEST['entryID'])) $this->entryID = intval($_REQUEST['entryID']);
		if (isset($_REQUEST['username'])) $this->username = StringUtil::trim($_REQUEST['username']);
		if (isset($_REQUEST['message'])) $this->message = StringUtil::trim($_REQUEST['message']);
		
		// check for commands
		if (preg_match("/\/[^[:space:]]+/", $this->message, $match)) {
			$this->commandName = StringUtil::replace('/', '', $match[0]);
		}	
	}
	
	/**
	 * @see Page::show()
	 */
	public function show() {
		parent::show();
		
		switch ($this->action) {
			case 'add':
				$this->addEntry();
				break;
			case 'delete':
				$this->deleteEntry();
				break;
			case 'getEntries':
				$this->getEntries();
				break;
			default:
				throw new IllegalLinkException();
		}
	}
	
	/**
	 * Adds a new shoutbox entry.
	 */
	protected function addEntry() {
		// check security token
		// i think it's a bit more secure if we check the security token, normally the security token
		// is checked automatically, but while submitting the entries there mustn't be a security token
		// because the shoutbox reads the entries every X seconds without a reload of the whole page so
		// the security token could be invalid.
		$this->checkSecurityToken();

		// currently there is no permission denied exception needed I think :)
		if (!WCF::getUser()->getPermission('user.shoutbox.canAddEntry')) return;

		try {
			// validate entry
			$this->validateEntry();
			
			// in this shoutbox version there is no command api formally, but you can use the
			// attribute 'commandName' to identify your wanted command, you can see it in the following
			// example (I know, it doesn't make so much sence, because there is already the name in front
			// of the message)
			
			// handle 'me'-command
			// if ($this->commandName == 'me') {
				// $this->message = StringUtil::replace('/me', WCF::getUser()->username);
			// }
		
			// save entry
			ShoutboxEntryEditor::create(WCF::getUser()->userID, $this->username, TIME_NOW, $this->message);
			
			// update activity points
			if (WCF::getUser()->userID && ACTIVITY_POINTS_PER_SHOUTBOX_ENTRY) {
				require_once(WCF_DIR.'lib/data/user/rank/UserRank.class.php');
				UserRank::updateActivityPoints(ACTIVITY_POINTS_PER_SHOUTBOX_ENTRY);
			}
		}
		catch (UserInputException $e) {
			// currently there is no error handling..
			// in the future errors will be shown. I think we do only need error handling while submitting
			// an entry because there we have user inputs..
			// the only possible error while deleting an entry is a system error (but that point handles
			// the woltlab community framework) and a permission denied exception (but it's not a user input
			// exception).
		}
	}
	
	/**
	 * Validates a new shoutbox entry.
	 */
	protected function validateEntry() {
		// username
		// only for guests
		if (WCF::getUser()->userID == 0) {
			// username
			if (empty($this->username)) {
				throw new UserInputException('username');
			}
			if (!UserUtil::isValidUsername($this->username)) {
				throw new UserInputException('username', 'notValid');
			}
			if (!UserUtil::isAvailableUsername($this->username)) {
				throw new UserInputException('username', 'notAvailable');
			}
			
			WCF::getSession()->setUsername($this->username);
		}
		else {
			$this->username = WCF::getUser()->username;
		}
		
		// message
		if (empty($this->message)) {
			throw new UserInputException('message');
		}
		if (StringUtil::length($this->message) > WCF::getUser()->getPermission('user.shoutbox.maxEntryLength')) {
			throw new UserInputException('message', 'tooLong');
		}
	}
	
	/**
	 * Deletes an entry.
	 */
	protected function deleteEntry() {
		// check security token
		$this->checkSecurityToken();

		// get shoutbox entry editor
		$entry = new ShoutboxEntryEditor($this->entryID);
		
		// check permission
		if ($entry->canDeleteEntry()) {
			$entry->delete();
		}
		else {
			// currently there is no response while deleting an entry. But if somebody calls the page directly
			// I think it's nice for him to know that he has no permission..
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * Reads entries and outputs them.
	 */
	protected function getEntries() {
		// get shoutbox entry list
		$entryList = new ShoutboxEntryList;
		$entryList->sqlOrderBy = 'time '.SHOUTBOX_ENTRY_SORT_ORDER;
		$entryList->sqlLimit = SHOUTBOX_MAX_ENTRIES;
		$entryList->readObjects();
		
		// assign variables
		WCF::getTPL()->assign(array(
			'shoutboxEntries' => $entryList->getObjects()
		));
			
		// send no cache headers
		HeaderUtil::sendNoCacheHeaders();
		
		// output template
		echo WCF::getTPL()->fetch('shoutboxEntries');
		exit;
	}
}
?>